pub enum Segment {
    Literal(&'static str),
    Argument(&'static str),
}

pub struct ConstRoute<T, U>
where
    T: 'static,
    U: 'static,
{
    pub method: &'static str,
    pub segments: &'static [Segment],
    pub wrapper: &'static dyn Fn(T, &[&str]) -> U,
}

pub fn router_core<T, U>(request: T, routes: &'static [&ConstRoute<T, U>]) -> Option<U>
where
    T: ::mimeograph_request::Request,
{
    let request_method: ::std::borrow::Cow<'_, str> =
        ::mimeograph_request::Request::method(&request);

    // create an array of Segments that can be matched
    let path = ::mimeograph_request::Request::path(&request).into_owned();
    let trimmed_path = path.trim_start_matches('/');
    let request_segments: Vec<&str> = trimmed_path.split('/').collect();

    'for_routes: for &route in routes {
        let ConstRoute {
            method: route_method,
            segments: route_segments,
            wrapper: route_wrapper,
        } = *route;

        // match the verb, returning None if there is no match
        if route_method != request_method {
            continue;
        }

        // match segments, returning None if there is no match
        if route_segments.len() != request_segments.len() {
            continue;
        }

        // Check the segments of the request, returning None if a match fails
        let mut args_count = 0;
        let segments_zip = route_segments.iter().zip(request_segments.iter());
        for (route_segment, request_segment) in segments_zip {
            match route_segment {
                Segment::Literal(seg_lit) => {
                    if request_segment != seg_lit {
                        continue 'for_routes;
                    }
                }
                Segment::Argument(_) => {
                    // skip the match
                    args_count += 1;
                }
            }
        }

        // We have a WINNER!
        // Collect the args
        let mut args = Vec::with_capacity(args_count);
        let segments_zip = route_segments.iter().zip(request_segments.iter());
        for (route_segment, request_segment) in segments_zip {
            match route_segment {
                Segment::Literal(_) => {}
                Segment::Argument(_) => {
                    // skip the match
                    // store the argument
                    args.push(*request_segment);
                }
            }
        }

        return Some(route_wrapper(request, &args));
    }

    return None;
}
