use crate::Result;

use heck::ShoutySnakeCase;
use proc_macro::TokenStream;
use proc_macro2::TokenStream as TokenStream2;
use quote::{quote, quote_spanned};
use syn::spanned::Spanned;
use syn::{parse::Parser, punctuated::Punctuated, token::Comma, Path};

pub(crate) fn wrap_last_segment(path: &mut Path, prefix: &str, suffix: &str) {
    let mut last_seg = path.segments.last_mut().expect("syn::Path has segments");
    last_seg.ident = quote::format_ident!(
        "{}{}{}",
        prefix,
        last_seg.ident.to_string().to_shouty_snake_case(),
        suffix
    );
}

fn wrapped_array(prefix: &str, suffix: &str, input: TokenStream) -> Result<TokenStream2> {
    // Parse a comma-separated list of paths.
    let mut paths = <Punctuated<Path, Comma>>::parse_terminated.parse(input)?;

    // Prefix the last segment in each path with `prefix`.
    paths
        .iter_mut()
        .for_each(|p| wrap_last_segment(p, prefix, suffix));

    // Return an `array` of the prefixed, mapped paths.
    let prefixed_mapped_paths = paths
        .iter()
        .map(|path| quote_spanned!(path.span().into() => &#path));

    Ok(quote!(&[#(#prefixed_mapped_paths),*]))
}

fn router_matcher(arr: TokenStream2) -> TokenStream2 {
    quote!(
        move |request| {
            ::mimeograph_router::router_core(request, #arr)
        }
    )
}

pub fn routes_macro(input: TokenStream) -> TokenStream {
    let arr = match wrapped_array(
        crate::ROUTE_STRUCT_PREFIX,
        crate::ROUTE_STRUCT_SUFFIX,
        input,
    ) {
        Ok(arr) => arr,
        Err(err) => {
            return err.into_compile_error().into();
        }
    };

    router_matcher(arr).into()
}
