use heck::ShoutySnakeCase;
use proc_macro::TokenStream;
use quote::quote;
use syn::parse_macro_input;
use syn::spanned::Spanned;

enum Segment {
    Glob,
    Wildcard,
    Literal(String),
    Argument(String),
}

type Segments = Vec<Segment>;

fn uri_path_to_segments(path: syn::LitStr) -> crate::Result<Segments> {
    let uri_path = path.value();

    if uri_path.is_empty() {
        return Err(syn::Error::new(path.span(), "route path may not be empty"));
    }

    if !matches!(uri_path.chars().next(), Some('/')) {
        return Err(syn::Error::new(path.span(), "routes must begin with a '/'"));
    }

    let segments: Segments = uri_path[1..]
        .split('/')
        .into_iter()
        .map(|s| {
            let mut chars = s.chars();

            match chars.next() {
                Some(':') => Ok(Segment::Argument(s[1..].to_owned())),
                // Might be:
                //  - Wildcard: /*/
                //  - Glob: /**/
                //  - Literal: /*foo/
                //  - Literal: /**bar/
                Some('*') => match chars.next() {
                    // Wildcard
                    None => Ok(Segment::Wildcard),
                    // Maybe glob
                    Some('*') => match chars.next() {
                        // Glob
                        None => Ok(Segment::Glob),
                        // Literal: /**something/
                        Some(_) => Ok(Segment::Literal(s.to_owned())),
                    },
                    // Literal: /*something/
                    Some(_) => Ok(Segment::Literal(s.to_owned())),
                },
                Some(_) => Ok(Segment::Literal(s.to_owned())),
                None => Ok(Segment::Literal(String::default())),
            }
        })
        .collect::<crate::Result<Segments>>()?;

    Ok(segments)
}

pub fn route_attribute(method: &'static str, args: TokenStream, input: TokenStream) -> TokenStream {
    let http_method = method.to_string().to_uppercase();
    let wrapped_fn = parse_macro_input!(input as syn::ItemFn);

    let attribute = parse_macro_input!(args as syn::LitStr);
    let segments = match uri_path_to_segments(attribute.clone()) {
        Ok(segments) => segments,
        Err(err) => {
            return err.into_compile_error().into();
        }
    };
    let segment_arg_count = segments
        .iter()
        .map(|s| match s {
            Segment::Glob | Segment::Wildcard | Segment::Literal(_) => 0_usize,
            Segment::Argument(_) => 1,
        })
        .sum();
    let glob_position = segments.iter().position(|s| matches!(s, Segment::Glob));

    // enforce that glob is last
    if let Some(position) = glob_position {
        if segments.len() - 1 != position {
            return syn::Error::new(attribute.span(), "'**' may only be at the end of a path")
                .into_compile_error()
                .into();
        }
    }
    let has_glob = glob_position.is_some();

    let struct_instance_name = quote::format_ident!(
        "{}{}{}",
        crate::ROUTE_STRUCT_PREFIX,
        wrapped_fn.sig.ident.to_string().to_shouty_snake_case(),
        crate::ROUTE_STRUCT_SUFFIX,
    );

    let wrapped_fn_name = quote::format_ident!("{}", wrapped_fn.sig.ident);
    let wrapper_fn_name = quote::format_ident!("wrapper_for_{}", wrapped_fn.sig.ident);

    // convert segments into TokenStreams
    let segments = segments.into_iter().map(|s| match s {
        Segment::Literal(lit) => {
            quote! { ::mimeograph_router::Segment::Literal(#lit) }
        }
        Segment::Argument(arg) => {
            quote! { ::mimeograph_router::Segment::Argument(#arg) }
        }
        Segment::Glob => {
            quote! { ::mimeograph_router::Segment::Glob }
        }
        Segment::Wildcard => {
            quote! { ::mimeograph_router::Segment::Wildcard }
        }
    });

    // the parameter list for the wrapped function
    let wrapped_args = (0..segment_arg_count).map(|x| quote!(args[#x]));

    let request_type = match wrapped_fn.sig.inputs.first().unwrap() {
        syn::FnArg::Receiver(recv) => {
            return syn::Error::new(recv.span(), "member functions not supported")
                .into_compile_error()
                .into();
        }
        syn::FnArg::Typed(typed) => {
            let ty = &typed.ty;
            quote!( #ty )
        }
    };

    let return_type = match &wrapped_fn.sig.output {
        syn::ReturnType::Default => quote! { () },
        syn::ReturnType::Type(_arrow, ty) => quote! { #ty },
    };

    let output = quote! {
        pub(crate) fn #wrapper_fn_name(request: #request_type, args: &[&str]) -> #return_type
        {
            // wrapped function
            #wrapped_fn

            // calling that function
            #wrapped_fn_name(request, #(#wrapped_args),*)
        }

        pub(crate) const #struct_instance_name: ::mimeograph_router::ConstRoute<#request_type, #return_type> = ::mimeograph_router::ConstRoute {
            method: #http_method,
            segments: &[#(#segments),*],
            wrapper: &#wrapper_fn_name,
            has_glob: #has_glob,
            args_count: #segment_arg_count,
        };
    };

    output.into()
}
