# A simple request router

Designed to be `wasm` compatible, simple and small.

```rust
use mimeograph_router::*;

/// A request could be an `http` crate Request, or a CloudFlare
/// worker Request. Here is a very simple request:
pub struct Request {
  path: String,
  method: String
}

/// To use the router, implement `mimeograph_request::Request`
/// for the Request type that you are working with.
impl mimeograph_request::Request for Request {
  fn path(&self) -> Cow<'_, str> {
    Cow::Borrowed(&self.path)
  }
  fn method(&self) -> Cow<'_, str> {
    Cow::Borrowed(&self.method)
  }
}

/// A dummy Response type
pub struct Response {
  status: u32,
  body: String
}

#[get("/hello")]
fn get_hello(request: Request) -> Result<Response> {
  Ok(Response { status: 200, body: String::from("world") })
}

#[get("/greet/:name")]
fn get_greeting(request: Request, name: &str) -> Result<Response> {
  Ok(Response { status: 200, body: format!("Hello, {}!", name) })
}

// Wildcard segment
#[get("/foo/*/baz/:name")]
fn foo_baz_name(request: Request, name: &str) -> Result<Response> {
  Ok(Response { status: 200, body: format!("Hello, {}!", name) })
}

// Glob, matches everything after /page/, ie: /page/hello/world
#[get("/page/**")]
fn page_glob(request: Request, name) -> Result<Response> {
  Ok(Response { status: 200, body: String::from("Hello, World") })
}

/// All verbs are supported:
/// get, put, post, delete, head, patch, options
#[post("/images")]
fn post_image(request: Request) -> Result<Response> {
  // handle request
  unimplemented!()
}

/// The entrypoint for handling a request
pub fn handle_request(request: Request) -> Response {
    // router![] creates a closure that iterates through
    // the handlers until one is matched, then it returns
    // the matched handler's response
    let router = router![get_hello, get_greeting, foo_baz_name, page_glob];

    match router(request) {
        // Matched a route
        Ok(Ok(resp)) => resp,

        // There was an error
        Ok(Err(_err)) => {
            // convert err into an http::Response;
            todo!()
        }

        // No routes matched, request is passed back
        Err(request) => {
            // provide a 404, redirect, or similar response
            todo!()
        }
    }
}
```
