use mimeograph_router::*;

type Result<T> = std::result::Result<T, Box<dyn std::error::Error>>;

#[get("/foo/bar/baz")]
pub fn foo_bar_baz(_request: http::Request<String>) -> Result<http::Response<String>> {
    http::Response::builder()
        .body(String::from("hello world"))
        .map_err(Into::into)
}

#[get("/bar/baz/:qux")]
pub fn bar_baz_qux(_request: http::Request<String>, qux: &str) -> Result<http::Response<String>> {
    http::Response::builder()
        .body(qux.to_owned())
        .map_err(Into::into)
}

#[get("/foo/*/bar/:qux")]
pub fn wild_foo_bar(_request: http::Request<String>, qux: &str) -> Result<http::Response<String>> {
    http::Response::builder()
        .body(qux.to_owned())
        .map_err(Into::into)
}

#[get("/foo/**")]
pub fn glob_foo(_request: http::Request<String>) -> Result<http::Response<String>> {
    http::Response::builder()
        .body(String::from("test"))
        .map_err(Into::into)
}

#[get("/**")]
pub fn glob(_request: http::Request<String>) -> Result<http::Response<String>> {
    http::Response::builder()
        .body(String::from("test"))
        .map_err(Into::into)
}

#[test]
fn match_get() {
    let request = http::Request::get("http://mimeograph.roosterbockets.com/foo/bar/baz")
        .body(String::new())
        .unwrap();

    // match the second route
    let router = router![bar_baz_qux, foo_bar_baz];

    let response = router(request);

    // should be Some(Ok(..))
    let body = response.expect("matched").unwrap().into_body();
    assert_eq!("hello world", body);
}

#[test]
fn match_get_with_arg() {
    let request = http::Request::get("http://mimeograph.roosterbockets.com/bar/baz/aoeu")
        .body(String::new())
        .unwrap();

    let router = router![bar_baz_qux, foo_bar_baz];

    let response = router(request);

    // should be Some(Ok(..))
    let body = response.expect("matched").unwrap().into_body();
    assert_eq!("aoeu", body);
}

#[test]
fn match_wildcard() {
    let request = http::Request::get("http://mimeograph.roosterbockets.com/foo/hello/bar/world")
        .body(String::new())
        .unwrap();

    let router = router![wild_foo_bar];

    let response = router(request);

    // should be Ok(Ok(..))
    let body = response.expect("matched").unwrap().into_body();
    assert_eq!("world", body);
}

#[test]
fn match_glob() {
    let request = http::Request::get("http://mimeograph.roosterbockets.com/this/is/some/test")
        .body(String::new())
        .unwrap();

    let router = router![glob];

    let response = router(request);

    // should be Ok(Ok(..))
    let body = response.expect("matched").unwrap().into_body();
    assert_eq!("test", body);
}

#[test]
fn match_glob_foo() {
    let request = http::Request::get("http://mimeograph.roosterbockets.com/foo/is/some/test")
        .body(String::new())
        .unwrap();

    let router = router![glob];

    let response = router(request);

    // should be Ok(Ok(..))
    let body = response.expect("matched").unwrap().into_body();
    assert_eq!("test", body);
}
