use crate::Result;

use proc_macro::TokenStream;
use proc_macro2::TokenStream as TokenStream2;
use quote::{quote, quote_spanned};
use syn::spanned::Spanned;
use syn::{parse::Parser, punctuated::Punctuated, token::Comma, Path};

pub(crate) fn prefix_last_segment(path: &mut Path, prefix: &str) {
    let mut last_seg = path.segments.last_mut().expect("syn::Path has segments");
    last_seg.ident = quote::format_ident!("{}{}", prefix, last_seg.ident);
}

fn _prefixed_array(prefix: &str, input: TokenStream) -> Result<(usize, TokenStream2)> {
    // Parse a comma-separated list of paths.
    let mut paths = <Punctuated<Path, Comma>>::parse_terminated.parse(input)?;

    // Prefix the last segment in each path with `prefix`.
    paths
        .iter_mut()
        .for_each(|p| prefix_last_segment(p, prefix));

    // Return an `array` of the prefixed, mapped paths.
    let prefixed_mapped_paths = paths
        .iter()
        .map(|path| quote_spanned!(path.span().into() => &#path));

    Ok((paths.len(), quote!([#(#prefixed_mapped_paths),*])))
}

fn prefixed_array(prefix: &str, input: TokenStream) -> Result<TokenStream2> {
    let (count, arr) = _prefixed_array(prefix, input)?;

    Ok(quote!({
        let __arr: [&dyn Fn(&[&str], _) -> _; #count] = #arr;
        core::array::IntoIter::new(__arr)
    }))
}

fn router_matcher(arr: TokenStream2) -> TokenStream2 {
    quote!(
        |request| -> Option<_> {
            // create an array of Segments that can be matched
            let path = ::mimeograph_request::Request::path(request);
            let trimmed_path = path.trim_start_matches('/');

            let segments: Vec<&str> = trimmed_path.split('/').collect();

            for route in #arr {
                match route(&segments, request) {
                    None => continue,
                    ok_or_error => {
                        return ok_or_error;
                    }
                }
            }
            None
        }
    )
}

pub fn routes_macro(input: TokenStream) -> TokenStream {
    let arr = match prefixed_array(crate::ROUTE_STRUCT_PREFIX, input) {
        Ok(arr) => arr,
        Err(err) => {
            return err.into_compile_error().into();
        }
    };

    router_matcher(arr).into()
}
