use crate::ROUTE_STRUCT_PREFIX;
use proc_macro::TokenStream;
use quote::quote;
use syn::parse_macro_input;
use syn::spanned::Spanned;

enum Segment {
    Literal(String),
    Argument(String),
}

type Segments = Vec<Segment>;

fn uri_path_to_segments(path: syn::LitStr) -> crate::Result<Segments> {
    let uri_path = path.value();

    if uri_path.is_empty() {
        return Err(syn::Error::new(path.span(), "routes must begin with a '/'"));
    }

    let segments: Segments = uri_path[1..]
        .split('/')
        .into_iter()
        .map(|s| match s.chars().next() {
            Some(':') => Ok(Segment::Argument(s[1..].to_owned())),
            Some(_) => Ok(Segment::Literal(s.to_owned())),
            None => Err(syn::Error::new(
                path.span(),
                "route segments may not be empty",
            )),
        })
        .collect::<crate::Result<Segments>>()?;

    Ok(segments)
}

pub fn route_attribute(method: &'static str, args: TokenStream, input: TokenStream) -> TokenStream {
    // let http_method_str = method.to_string().to_lowercase();

    let wrapped_fn = parse_macro_input!(input as syn::ItemFn);
    let attribute = parse_macro_input!(args as syn::LitStr);
    let segments = match uri_path_to_segments(attribute) {
        Ok(segments) => segments,
        Err(err) => {
            return err.into_compile_error().into();
        }
    };
    let segment_count = segments.len();
    let segment_arg_count = segments
        .iter()
        .filter(|s| matches!(s, Segment::Argument(_)))
        .count();

    // build up the wrapping function name, ie: FOO_BAR_THE_WRAPPED_FN
    let wrapper_name = quote::format_ident!(
        "{}{}",
        ROUTE_STRUCT_PREFIX,
        wrapped_fn.sig.ident.to_string()
    );

    let wrapped_fn_name = quote::format_ident!("{}", wrapped_fn.sig.ident);
    let wrapped_fn_input = match wrapped_fn.sig.inputs.first().unwrap() {
        syn::FnArg::Receiver(recv) => {
            return syn::Error::new(recv.span(), "member functions not supported")
                .into_compile_error()
                .into();
        }
        syn::FnArg::Typed(typed) => {
            let ty = &typed.ty;
            quote!( #ty )
        }
    };

    // convert segments into TokenStreams
    let segments: Vec<_> = segments
        .into_iter()
        .map(|s| match s {
            Segment::Literal(lit) => {
                quote! { Segment::Literal(#lit) }
            }
            Segment::Argument(arg) => {
                quote! { Segment::Argument(#arg) }
            }
        })
        .collect();

    // the parameter list for the wrapped function
    let wrapped_args = (0..segment_arg_count).map(|x| quote!(args[#x]));
    let wrapped_args_default = (0..segment_arg_count).map(|_x| quote!(""));

    let return_type = match &wrapped_fn.sig.output {
        syn::ReturnType::Default => quote! { () },
        syn::ReturnType::Type(_arrow, ty) => quote! { #ty },
    };

    let output = quote! {
        pub fn #wrapper_name(
            route_segments: &[&str],
            request: #wrapped_fn_input,
        ) -> Option<#return_type> {
            #[allow(dead_code)]
            enum Segment {
                Literal(&'static str),
                Argument(&'static str),
            }

            let segments: [Segment; #segment_count] = [#(#segments),*];

            // match the verb, returning None if there is no match
            if #method != ::mimeograph_request::Request::method(request) {
                return None;
            }

            // match segments, returning None if there is no match
            if segments.len() != route_segments.len() {
                return None;
            }

            // collect the args into this array
            let mut args: [&str; #segment_arg_count] = [#(#wrapped_args_default),*];
            let mut args_index = 0;

            let segments_zip = core::array::IntoIter::new(segments)
                .zip(route_segments.iter());
            for (segment, &route_segment) in segments_zip {
                match segment {
                    Segment::Literal(seg_lit) => {
                        if route_segment != seg_lit {
                            return None;
                        }
                    },
                    Segment::Argument(seg_arg) => {
                        // store the argument
                        args[args_index] = route_segment;
                        args_index += 1;
                    },
                }
            }

            // call wrapped function
            Some(#wrapped_fn_name(request, #(#wrapped_args),*))
        }
        #wrapped_fn
    };

    output.into()
}
