extern crate alloc;

mod durable_objects;
mod error;
mod html_rewriter;
mod kv;

pub use durable_objects::*;
pub use error::*;
pub use html_rewriter::*;
use js_sys::Boolean;
use js_sys::Uint8Array;
pub use kv::*;

use alloc::vec::Vec;

use wasm_bindgen::prelude::*;
use wasm_bindgen::JsCast;
use wasm_bindgen_futures::JsFuture;

use web_sys::ReadableStreamGetReaderOptions;
use web_sys::ReadableStreamReaderMode;
pub use web_sys::{
    Headers, ReadableStream, ReadableStreamByobReadResult, ReadableStreamByobReader,
    ReadableStreamDefaultReadResult, ReadableStreamDefaultReader, Request, RequestInit, Response,
    ResponseInit, TextDecoder, TextEncoder, TransformStream, WritableStream,
    WritableStreamDefaultWriter,
};

pub type Result<T> = core::result::Result<T, Error>;

pub trait ObjectProperty {
    fn property<T: JsCast>(&self, name: &str) -> Option<T>;
}

impl ObjectProperty for js_sys::Object {
    fn property<T: JsCast>(&self, key: &str) -> Option<T> {
        let key = js_sys::JsString::from(key);

        match js_sys::Reflect::get(&self, &key) {
            Ok(value) => {
                if value.is_undefined() {
                    return None;
                }
                value.dyn_into::<T>().ok()
            }
            Err(_) => unreachable!("Object::property"),
        }
    }
}

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen(js_namespace=crypto, js_name=subtle)]
    pub static SUBTLE_CRYPTO: web_sys::SubtleCrypto;

    #[wasm_bindgen(js_name=crypto)]
    pub static CRYPTO: web_sys::Crypto;

    #[wasm_bindgen(catch)]
    pub fn btoa(btoa: &js_sys::JsString) -> core::result::Result<String, JsValue>;

    #[wasm_bindgen(catch)]
    pub fn atob(btoa: &str) -> core::result::Result<js_sys::JsString, JsValue>;
}

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen]
    #[derive(Clone)]
    pub type Event;

    #[wasm_bindgen(getter, method)]
    pub fn request(this: &Event) -> Request;

    #[wasm_bindgen(getter, method, js_name = type)]
    pub fn kind(this: &Event) -> String;
}

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen]
    #[derive(Clone)]
    pub type Fetcher;
}

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen]
    #[derive(Clone)]
    pub type Env;
}

pub async fn read_body(
    body_stream: &ReadableStream,
    max_size: Option<u32>,
) -> crate::Result<Vec<u8>> {
    let body_reader = body_stream
        .get_reader()
        .unchecked_into::<ReadableStreamDefaultReader>();

    let mut buffer = Vec::new();
    let mut position = 0_u32;

    loop {
        let result = JsFuture::from(body_reader.read())
            .await
            .map(|v| v.unchecked_into::<ReadableStreamDefaultReadResult>())
            .map_err(crate::Error::from)?;

        if let Some(value) = result.property::<Boolean>("done") {
            if value.value_of() {
                break;
            }
        }

        // check if done before reading value, it could be undefined
        let arr = Uint8Array::new(&result.property("value").unwrap());
        let length = arr.byte_length() as usize;
        let end = position as usize + length;

        if let Some(max_size) = max_size {
            if end > max_size as usize {
                return Err(crate::Error::from("request body too big"));
            }
        }

        buffer.resize(end, 0);
        arr.copy_to(&mut buffer[position as usize..(position as usize + length)]);
        position += length as u32;
    }

    Ok(buffer)
}

pub trait RestOk: Sized {
    /// Converts the response into
    fn rest_ok(self) -> crate::Result<Self>;
}

impl RestOk for web_sys::Response {
    fn rest_ok(self) -> crate::Result<Self> {
        match self.status() {
            200..=299 | 404 => Ok(self),
            _ => Err(crate::Error::from_rest(self)),
        }
    }
}
pub fn get_byob_reader(this: &ReadableStream) -> ReadableStreamByobReader {
    let mut options = ReadableStreamGetReaderOptions::new();
    options.mode(ReadableStreamReaderMode::Byob);
    this.get_reader_with_options(&options).unchecked_into()
}
