use alloc::string::String;
use js_sys::{ArrayBuffer, Promise};
use wasm_bindgen::prelude::*;

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen]
    #[derive(Clone)]
    pub type KV;

    /// The promise returns a Result<T, E> where T is a kind defined in KVGetOptions
    #[wasm_bindgen(method)]
    pub fn get(this: &KV, key: &str) -> Promise;

    #[wasm_bindgen(method, js_name=get)]
    pub fn get_with_options(this: &KV, key: &str, options: KVGetOptions) -> Promise;

    /// returns a Promise<MetadataValue>
    #[wasm_bindgen(method, js_name=getWithMetadata)]
    pub fn get_with_metadata(this: &KV, key: &str) -> Promise;

    /// returns a Promise<MetadataValue>
    #[wasm_bindgen(method, js_name=getWithMetadata)]
    pub fn get_with_options_and_metadata(this: &KV, key: &str, options: KVGetOptions) -> Promise;

    #[wasm_bindgen(method, js_name=put)]
    pub fn put_string(this: &KV, key: &str, value: &str) -> Promise;

    #[wasm_bindgen(method, js_name=put)]
    pub fn put_stream(this: &KV, key: &str, value: super::ReadableStream) -> Promise;

    #[wasm_bindgen(method, js_name=put)]
    pub fn put_buffer(this: &KV, key: &str, value: ArrayBuffer) -> Promise;

    #[wasm_bindgen(method)]
    pub async fn delete(this: &KV, key: &str) -> JsValue;
}

#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen(js_name=Object)]
    #[derive(Clone)]
    pub type MetadataValue;

    /// `value` is null or the request type from KV.get() or KV.getWithMetadata()
    #[wasm_bindgen(getter, method, js_class=Object)]
    pub fn value(this: &MetadataValue) -> JsValue;

    /// `metadata` is null or a `String`
    #[wasm_bindgen(getter, method, js_class=Object)]
    pub fn metadata(this: &MetadataValue) -> JsValue;
}

pub enum GetKind {
    Text,
    Json,
    ArrayBuffer,
    Stream,
}

impl From<GetKind> for KVGetOptions {
    fn from(get_kind: GetKind) -> Self {
        Self { kind: get_kind }
    }
}

#[wasm_bindgen]
pub struct KVGetOptions {
    kind: GetKind,
}

#[wasm_bindgen]
impl KVGetOptions {
    #[wasm_bindgen(getter, method, js_name=type)]
    pub fn kind(&self) -> String {
        match self.kind {
            GetKind::Text => String::from("text"),
            GetKind::Json => String::from("json"),
            GetKind::ArrayBuffer => String::from("arrayBuffer"),
            GetKind::Stream => String::from("stream"),
        }
    }
}
