use alloc::borrow::Cow;
use alloc::string::String;
use crate::alloc::borrow::ToOwned;

#[derive(Debug)]
pub enum Error {
    Generic(Cow<'static, str>),
    Http(u16, String),
    Rest(u16, String),
    JsError(js_sys::Error),
}

impl Error {
    pub fn from_http(response: crate::Response) -> Self {
        Self::Http(response.status(), response.status_text())
    }

    pub fn from_rest(response: crate::Response) -> Self {
        Self::Rest(response.status(), response.status_text())
    }
}

impl core::fmt::Display for Error {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        match self {
            Error::Generic(message) => f.write_str(&message),
            Error::Http(status, message) => write!(f, "HTTP Error {}, {}", status, message),
            Error::Rest(status, message) => write!(f, "REST Error {}, {}", status, message),
            Error::JsError(error) => {
                let message = String::from(error.message());
                write!(f, "JavaScript Error: {}", message)
            }
        }
    }
}

impl From<&str> for Error {
    fn from(message: &str) -> Self {
        Self::Generic(Cow::from(message.to_owned()))
    }
}

impl From<String> for Error {
    fn from(message: String) -> Self {
        Self::Generic(Cow::from(message))
    }
}

impl From<wasm_bindgen::JsValue> for Error {
    fn from(err: wasm_bindgen::JsValue) -> Self {
        let err = js_sys::Error::from(err);
        Self::JsError(err)
    }
}
