use js_sys::{ArrayBuffer, Promise, Uint8Array};
use wasm_bindgen::prelude::*;

#[wasm_bindgen]
extern "C" {
    #[derive(Clone)]
    pub type ReadableStream;

    #[wasm_bindgen(method, js_name=pipeTo)]
    pub fn pipe_to(this: &ReadableStream, destination: &WritableStream) -> Promise;

    #[wasm_bindgen(method, js_name=pipeTo)]
    pub fn pipe_to_with_options(
        this: &ReadableStream,
        destination: &WritableStream,
        options: PipeToOptions,
    ) -> Promise;

    #[wasm_bindgen(method, js_name=getReader)]
    pub fn get_reader(this: &ReadableStream) -> ReadableStreamDefaultReader;

    #[wasm_bindgen(method, js_name=getReader)]
    pub fn get_reader_with_options(
        this: &ReadableStream,
        options: Option<BYOBOption>,
    ) -> ReadableStreamBYOBReader;
}

#[wasm_bindgen]
extern "C" {
    pub type WritableStreamDefaultWriter;

    #[wasm_bindgen(getter, method)]
    pub fn desiredSize(this: &WritableStreamDefaultWriter) -> i32;

    #[wasm_bindgen(getter, method)]
    pub fn closed(this: &WritableStreamDefaultWriter) -> Promise;

    #[wasm_bindgen(method)]
    pub fn abort(this: &WritableStreamDefaultWriter, reason: Option<&str>) -> Promise;

    #[wasm_bindgen(method)]
    pub fn close(this: &WritableStreamDefaultWriter) -> Promise;

    #[wasm_bindgen(method, js_name=releaseLock)]
    pub fn release_lock(this: &WritableStreamDefaultWriter) -> Promise;

    #[wasm_bindgen(method)]
    pub fn write(this: &WritableStreamDefaultWriter, data: &Uint8Array) -> Promise;
}

#[wasm_bindgen]
extern "C" {
    pub type WritableStream;

    #[wasm_bindgen(getter, method)]
    pub fn locked(this: &WritableStream) -> bool;

    #[wasm_bindgen(method, js_name=getWriter)]
    pub fn get_writer(this: &WritableStream) -> WritableStreamDefaultWriter;

    #[wasm_bindgen(method)]
    pub fn abort(this: &WritableStream, reason: Option<&str>) -> Promise;
}

#[wasm_bindgen]
extern "C" {
    pub type ReadableStreamDefaultReader;

    #[wasm_bindgen(method, js_name=read)]
    pub fn read(this: &ReadableStreamDefaultReader) -> Promise;
}

#[wasm_bindgen]
extern "C" {
    pub type ReadableStreamBYOBReader;

    /// Returns Promise<ReadableStreamReadResult>
    #[wasm_bindgen(method, js_name=read)]
    pub fn read_u8(this: &ReadableStreamBYOBReader, buffer: &Uint8Array) -> Promise;
}

#[wasm_bindgen]
extern "C" {
    pub type ReadableStreamReadResult;

    #[wasm_bindgen(getter, method)]
    pub fn value(this: &ReadableStreamReadResult) -> ArrayBuffer;

    #[wasm_bindgen(getter, method)]
    pub fn done(this: &ReadableStreamReadResult) -> bool;
}

#[wasm_bindgen]
extern "C" {
    pub type TransformStream;

    #[wasm_bindgen(constructor)]
    pub fn new() -> TransformStream;

    #[wasm_bindgen(getter, method)]
    pub fn readable(this: &TransformStream) -> ReadableStream;

    #[wasm_bindgen(getter, method)]
    pub fn writable(this: &TransformStream) -> WritableStream;
}

#[wasm_bindgen]
extern "C" {
    pub type TextDecoder;

    #[wasm_bindgen(constructor)]
    pub fn new() -> TextDecoder;

    #[wasm_bindgen(getter, method)]
    pub fn readable(this: &TextDecoder) -> ReadableStream;

    #[wasm_bindgen(getter, method)]
    pub fn writable(this: &TextDecoder) -> WritableStream;

    #[wasm_bindgen(method)]
    pub fn decode(this: &TextDecoder, data: &ArrayBuffer) -> String;
}

#[wasm_bindgen]
extern "C" {
    pub type TextEncoder;

    #[wasm_bindgen(constructor)]
    pub fn new() -> TextEncoder;

    #[wasm_bindgen(getter, method)]
    pub fn encoding(this: &TextEncoder) -> String;

    #[wasm_bindgen(method)]
    pub fn encode(this: &TextEncoder, input: &str) -> Uint8Array;
}

#[wasm_bindgen]
pub struct PipeToOptions {
    #[wasm_bindgen(readonly, js_name=preventClose)]
    pub prevent_close: bool,

    #[wasm_bindgen(readonly, js_name=preventAbort)]
    pub prevent_abort: bool,
}

impl Default for PipeToOptions {
    fn default() -> Self {
        Self {
            prevent_abort: false,
            prevent_close: false,
        }
    }
}

#[wasm_bindgen]
pub struct BYOBOption {}

#[wasm_bindgen]
impl BYOBOption {
    #[wasm_bindgen(getter, method, js_name=mode)]
    pub fn mode(&self) -> js_sys::JsString {
        js_sys::JsString::from("byob")
    }
}
