use js_sys::Promise;
use wasm_bindgen::prelude::*;

use super::Headers;

#[wasm_bindgen]
extern "C" {
    pub type Response;

    #[wasm_bindgen(constructor)]
    pub fn new(body: super::ReadableStream, options: ResponseOptions) -> Response;

    #[wasm_bindgen(constructor)]
    pub fn new_string(body: &str, options: ResponseOptions) -> Response;

    #[wasm_bindgen(setter, method, js_name = encodeBody)]
    pub fn set_encode_body(this: &Response, val: &str);

    #[wasm_bindgen(setter, method)]
    pub fn set_headers(this: &Response, val: super::Headers);

    #[wasm_bindgen(setter, method)]
    pub fn set_ok(this: &Response, val: bool);

    #[wasm_bindgen(setter, method)]
    pub fn set_redirected(this: &Response, val: bool);

    #[wasm_bindgen(setter, method)]
    pub fn set_status(this: &Response, val: u32);

    #[wasm_bindgen(setter, method, js_name = statusText)]
    pub fn set_status_text(this: &Response, val: &str);

    #[wasm_bindgen(setter, method)]
    pub fn set_url(this: &Response, val: &str);

    #[wasm_bindgen(getter, method)]
    pub fn body(this: &Response) -> super::ReadableStream;

    #[wasm_bindgen(getter, method, js_name=arrayBuffer)]
    pub fn array_buffer(this: &Response) -> Promise;

    #[wasm_bindgen(getter, method, js_name=formData)]
    pub fn form_data(this: &Response) -> Promise;

    #[wasm_bindgen(getter, method)]
    pub fn json(this: &Response) -> Promise;

    #[wasm_bindgen(getter, method)]
    pub fn text(this: &Response) -> Promise;

    #[wasm_bindgen(getter, method, js_name=bodyUsed)]
    pub fn body_used(this: &Response) -> bool;

    #[wasm_bindgen(getter, method, js_name=encodeBody)]
    pub fn encode_body(this: &Response) -> String;

    #[wasm_bindgen(getter, method)]
    pub fn headers(this: &Response) -> Headers;

    #[wasm_bindgen(getter, method)]
    pub fn ok(this: &Response) -> bool;

    #[wasm_bindgen(getter, method)]
    pub fn redirected(this: &Response) -> bool;

    #[wasm_bindgen(getter, method)]
    pub fn status(this: &Response) -> u32;

    #[wasm_bindgen(getter, method, js_name=statusText)]
    pub fn status_text(this: &Response) -> String;

    #[wasm_bindgen(getter, method)]
    pub fn url(this: &Response) -> String;
}

#[wasm_bindgen]
pub struct ResponseOptions {
    #[wasm_bindgen(readonly)]
    pub status: u32,

    // FIXME: Does this do what we think it does?
    // Will None be `undefined`?
    #[wasm_bindgen(readonly)]
    pub status_text: Option<&'static str>,
}

pub trait RestOk: Sized {
    /// Converts the response into
    fn rest_ok(self) -> crate::Result<Self>;
}

impl RestOk for Response {
    fn rest_ok(self) -> crate::Result<Self> {
        match self.status() {
            200..=299 | 404 => Ok(self),
            _ => Err(crate::Error::from_rest(self)),
        }
    }
}
