use js_sys::Uint8Array;
use wasm_bindgen::{prelude::*, JsCast};
use wasm_bindgen_futures::JsFuture;

use super::{Headers, ReadableStream};

#[wasm_bindgen]
extern "C" {
    pub type Request;

    #[wasm_bindgen(constructor)]
    pub fn new_with_request(request: &Request, options: RequestInit) -> Request;

    #[wasm_bindgen(constructor)]
    pub fn new_with_path(path: &str, options: RequestInit) -> Request;

    #[wasm_bindgen(constructor)]
    pub fn new(path: &str) -> Request;

    #[wasm_bindgen(getter, method)]
    pub fn cf(this: &Request) -> Cf;
    #[wasm_bindgen(getter, method)]
    pub fn fetcher(this: &Request) -> super::Fetcher;
    #[wasm_bindgen(getter, method)]
    pub fn redirect(this: &Request) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn headers(this: &Request) -> super::Headers;
    #[wasm_bindgen(getter, method)]
    pub fn url(this: &Request) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn method(this: &Request) -> String;
    #[wasm_bindgen(getter, method, js_name=bodyUsed)]
    pub fn body_used(this: &Request) -> bool;
    #[wasm_bindgen(getter, method)]
    pub fn body(this: &Request) -> super::ReadableStream;
}

#[wasm_bindgen]
extern "C" {
    pub type Cf;

    #[wasm_bindgen(getter, method)]
    pub fn asn(this: &Cf) -> i32;
    #[wasm_bindgen(getter, method)]
    pub fn city(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn clientAcceptEncoding(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn clientTcpRtt(this: &Cf) -> i32;
    #[wasm_bindgen(getter, method)]
    pub fn colo(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn continent(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn country(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn edgeRequestKeepAliveStatus(this: &Cf) -> i32;
    #[wasm_bindgen(getter, method)]
    pub fn httpProtocol(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn latitude(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn longitude(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn metroCode(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn postalCode(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn region(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn regionCode(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn requestPriority(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn timezone(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn tlsCipher(this: &Cf) -> String;
    #[wasm_bindgen(getter, method)]
    pub fn tlsClientAuth(this: &Cf) -> TlsClientAuth;
    #[wasm_bindgen(getter, method)]
    pub fn tlsExportedAuthenticator(this: &Cf) -> TlsExportedAuthenticator;
    #[wasm_bindgen(getter, method)]
    pub fn tlsVersion(this: &Cf) -> String;
}

#[wasm_bindgen]
extern "C" {
    pub type TlsClientAuth;
}

#[wasm_bindgen]
extern "C" {
    pub type TlsExportedAuthenticator;
}

pub enum RequestInitRedirect {
    Follow,
    Error,
    Manual,
}

#[wasm_bindgen]
#[derive(Default)]
pub struct RequestInit {
    // cf: Option<()>,
    method: Option<String>,
    headers: Option<Headers>,
    body_text: Option<String>,
    body_stream: Option<ReadableStream>,
    redirect: Option<RequestInitRedirect>,
}

impl RequestInit {
    pub fn new() -> Self {
        Self::default()
    }
    pub fn set_method(&mut self, value: &str) {
        self.method.replace(value.to_owned());
    }
    pub fn set_headers(&mut self, value: Headers) {
        self.headers.replace(value);
    }
    pub fn set_body_text(&mut self, value: &str) {
        self.body_text.replace(value.to_owned());
    }
    pub fn set_body_stream(&mut self, value: ReadableStream) {
        self.body_stream.replace(value);
    }
    pub fn set_redirect(&mut self, value: RequestInitRedirect) {
        self.redirect.replace(value);
    }
}

#[wasm_bindgen]
impl RequestInit {
    // #[wasm_bindgen(getter)]
    // pub fn cf(&self) -> JsValue {
    //     match self.cf {
    //         Some(cf) => cf.unchecked_ref::<JsValue>(),
    //         None => JsValue::undefined(),
    //     }
    // }

    #[wasm_bindgen(getter)]
    pub fn method(&self) -> JsValue {
        match &self.method {
            Some(method) => JsValue::from_str(&method),
            None => JsValue::undefined(),
        }
    }

    #[wasm_bindgen(getter)]
    pub fn headers(&self) -> JsValue {
        match &self.headers {
            Some(headers) => headers.clone().unchecked_into::<JsValue>(),
            None => JsValue::undefined(),
        }
    }

    #[wasm_bindgen(getter)]
    pub fn body(&self) -> JsValue {
        if let Some(stream) = &self.body_stream {
            return stream.clone().unchecked_into::<JsValue>();
        }
        match &self.body_text {
            Some(body_text) => JsValue::from_str(&body_text),
            None => JsValue::undefined(),
        }
    }

    #[wasm_bindgen(getter)]
    pub fn redirect(&self) -> JsValue {
        match self.redirect {
            Some(RequestInitRedirect::Manual) => JsValue::from_str("manual"),
            Some(RequestInitRedirect::Error) => JsValue::from_str("error"),
            Some(RequestInitRedirect::Follow) => JsValue::from_str("follow"),
            None => JsValue::undefined(),
        }
    }
}

pub async fn read_body(request: &Request, max_size: u32) -> crate::Result<Vec<u8>> {
    let body_stream = request.body();
    let body_reader = body_stream.get_reader();

    let mut buffer = Vec::with_capacity(max_size as usize);
    let mut position = 0_u32;

    loop {
        let result = JsFuture::from(body_reader.read())
            .await
            .map(|v| v.unchecked_into::<super::ReadableStreamReadResult>())
            .map_err(crate::Error::from)?;

        if result.done() {
            break;
        }

        // check if done before reading value, it could be undefined
        let arr = Uint8Array::new(&result.value());
        let length = arr.byte_length() as usize;
        let end = position as usize + length;

        if end > max_size as usize {
            return Err(crate::Error::from("request body too big"));
        }

        buffer.resize(end, 0);
        arr.copy_to(&mut buffer[position as usize..(position as usize + length)]);
        position += length as u32;
    }

    Ok(buffer)
}
