use wasm_bindgen::prelude::*;

use crate::Response;

#[wasm_bindgen]
extern "C" {
    pub type HTMLRewriter;

    #[wasm_bindgen(constructor)]
    pub fn new() -> HTMLRewriter;

    #[wasm_bindgen(method)]
    pub fn on(this: &HTMLRewriter, selector: &str, handler: CfElementHandler) -> HTMLRewriter;

    #[wasm_bindgen(method, js_name=onDocument)]
    pub fn on_document(this: &HTMLRewriter, handler: CfDocumentHandler) -> HTMLRewriter;

    /// It's unclear what the content and return types are!
    #[wasm_bindgen(method)]
    pub fn transform(this: &HTMLRewriter, content: Response) -> Response;

}

pub trait ElementHandler {
    /// Can return a Promise, or undefined
    fn element(&self, element: Element) -> JsValue;

    /// Can return a Promise, or undefined
    fn comments(&self, comment: Comment) -> JsValue;

    /// Can return a Promise, or undefined
    fn text(&self, text: Text) -> JsValue;
}

#[wasm_bindgen]
pub struct CfElementHandler(Box<dyn ElementHandler>);

impl CfElementHandler {
    pub fn new<T>(handler: Box<T>) -> Self
    where
        T: ElementHandler + 'static,
    {
        Self(handler)
    }
}
#[wasm_bindgen]
impl CfElementHandler {
    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn element(&self, element: Element) -> JsValue {
        self.0.element(element)
    }

    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn comments(&self, comment: Comment) -> JsValue {
        self.0.comments(comment)
    }

    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn text(&self, text: Text) -> JsValue {
        self.0.text(text)
    }
}

pub trait DocumentHandler {
    /// Can return a Promise, or undefined
    fn doctype(&self, doctype: Doctype) -> JsValue;

    /// Can return a Promise, or undefined
    fn comments(&self, comment: Comment) -> JsValue;

    /// Can return a Promise, or undefined
    fn text(&self, text: Text) -> JsValue;

    /// Can return a Promise, or undefined
    fn end(&self, end: End) -> JsValue;
}

#[wasm_bindgen]
pub struct CfDocumentHandler(Box<dyn DocumentHandler>);

impl CfDocumentHandler {
    pub fn new<T>(handler: T) -> Self
    where
        T: Into<Box<dyn DocumentHandler>>,
    {
        Self(handler.into())
    }
}

#[wasm_bindgen]
impl CfDocumentHandler {
    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn doctype(&self, doctype: Doctype) -> JsValue {
        self.0.doctype(doctype)
    }

    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn comments(&self, comment: Comment) -> JsValue {
        self.0.comments(comment)
    }

    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn text(&self, text: Text) -> JsValue {
        self.0.text(text)
    }

    /// Can return a Promise, or undefined
    #[wasm_bindgen]
    pub fn end(&self, end: End) -> JsValue {
        self.0.end(end)
    }
}

#[wasm_bindgen]
extern "C" {
    pub type Element;

    #[wasm_bindgen(method, getter, js_name=tagName)]
    pub fn tag_name(this: &Element) -> String;

    #[wasm_bindgen(method, getter)]
    pub fn attributes(this: &Element) -> js_sys::Iterator;

    #[wasm_bindgen(method, getter)]
    pub fn removed(this: &Element) -> bool;

    #[wasm_bindgen(method, getter, js_name=namespaceURI)]
    pub fn namespace_uri(this: &Element) -> String;

    #[wasm_bindgen(method)]
    pub fn getAttribute(this: &Element, name: &str) -> String;
    #[wasm_bindgen(method)]
    pub fn hasAttribute(this: &Element, name: &str) -> bool;
    #[wasm_bindgen(method)]
    pub fn setAttribute(this: &Element, name: &str, value: &str) -> Element;
    #[wasm_bindgen(method)]
    pub fn removeAttribute(this: &Element, name: &str) -> Element;
    #[wasm_bindgen(method)]
    pub fn before(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn after(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn prepend(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn append(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn replace(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn setInnerContent(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn remove(this: &Element, content: &str, escape: Escape) -> Element;
    #[wasm_bindgen(method)]
    pub fn removeAndKeepContent(this: &Element, content: &str, escape: Escape) -> Element;
}

#[wasm_bindgen]
extern "C" {
    pub type Text;

    // Properties
    #[wasm_bindgen(method, getter)]
    pub fn removed(this: &Text) -> bool;
    #[wasm_bindgen(method, getter)]
    pub fn text(this: &Text) -> String;
    #[wasm_bindgen(method, getter, js_name=lastInTextNode)]
    pub fn last_in_text_node(this: &Text) -> bool;

    // Methods
    #[wasm_bindgen(method)]
    pub fn before(this: &Text, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn after(this: &Text, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn replace(this: &Text, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn remove(this: &Text);
}

#[wasm_bindgen]
extern "C" {
    pub type Doctype;

    #[wasm_bindgen(method, getter)]
    pub fn name(this: &Doctype) -> String;

    #[wasm_bindgen(method, getter)]
    pub fn publicId(this: &Doctype) -> String;

    #[wasm_bindgen(method, getter)]
    pub fn systemId(this: &Doctype) -> String;
}
#[wasm_bindgen]
extern "C" {
    pub type Comment;

    #[wasm_bindgen(method, getter)]
    pub fn removed(this: &Comment) -> bool;

    #[wasm_bindgen(method, getter)]
    pub fn text(this: &Comment) -> String;

    #[wasm_bindgen(method)]
    pub fn before(this: &Comment, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn after(this: &Comment, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn replace(this: &Comment, content: &str, escape: Escape);
    #[wasm_bindgen(method)]
    pub fn remove(this: &Comment);
}

#[wasm_bindgen]
extern "C" {
    pub type End;

    #[wasm_bindgen(method)]
    pub fn append(this: &End, content: &str, escape: Escape);
}

#[wasm_bindgen]
pub struct Escape(pub bool);

#[wasm_bindgen]
impl Escape {
    #[wasm_bindgen(getter, method)]
    pub fn html(&self) -> bool {
        // https://developers.cloudflare.com/workers/runtime-apis/html-rewriter#global-types
        // true = no escaping
        !self.0
    }
}
