//! Common imports throughout this project.

pub(crate) use std::{
    collections::{BTreeMap as Map, BTreeSet as Set},
    fmt,
    io::Write,
    ops::{Deref, DerefMut},
};

pub use either::Either;
pub use error_chain::bail;
pub use num::{bigint::Sign, BigInt as Int, BigRational as Rat, One, Zero};
pub use rsmt2::SmtRes;

pub use crate::{
    build_decls, build_expr, build_trans, build_typ, check,
    err::*,
    expr::{self, HasTyp},
    parse, trans,
};

/// Step index.
///
/// In the context of a stateful expression, this is the index of the *current step*. If this index
/// is `7` for instance, then state variable `v` in the current step will be `v_7` and will be `v_8`
/// in the next step.
pub type Unroll = usize;

/// Style trait.
#[cfg(feature = "parser")]
pub trait Style {
    /// Output generated by styling functions.
    type Styled: fmt::Display;
    /// Bold styling.
    fn bold(&self, s: &str) -> Self::Styled;
    /// Red styling.
    fn red(&self, s: &str) -> Self::Styled;
    /// Green styling.
    fn green(&self, s: &str) -> Self::Styled;
    /// Underline styling.
    fn under(&self, s: &str) -> Self::Styled;
}

#[cfg(feature = "parser")]
impl Style for () {
    type Styled = String;
    fn bold(&self, s: &str) -> String {
        s.into()
    }
    fn red(&self, s: &str) -> String {
        s.into()
    }
    fn green(&self, s: &str) -> String {
        s.into()
    }
    fn under(&self, s: &str) -> String {
        s.into()
    }
}
#[cfg(feature = "parser")]
impl<'a, T> Style for &'a T
where
    T: Style,
{
    type Styled = T::Styled;
    fn bold(&self, s: &str) -> Self::Styled {
        (*self).bold(s)
    }
    fn red(&self, s: &str) -> Self::Styled {
        (*self).red(s)
    }
    fn green(&self, s: &str) -> Self::Styled {
        (*self).green(s)
    }
    fn under(&self, s: &str) -> Self::Styled {
        (*self).under(s)
    }
}
