use crate::{
    BaseElement, FieldElement, BASE_CYCLE_LENGTH as NUM_ROUNDS, OP_SPONGE_WIDTH,
    PROGRAM_DIGEST_SIZE,
};

// PUBLIC CONSTANTS
// ================================================================================================

pub const STATE_WIDTH: usize = OP_SPONGE_WIDTH;
pub const DIGEST_SIZE: usize = PROGRAM_DIGEST_SIZE;

// ACCUMULATOR FUNCTIONS
// ================================================================================================

/// Executes a modified version of [Rescue](https://eprint.iacr.org/2019/426) round where inputs
/// are injected into the sate in the middle of the round. This modification differs significantly
/// form how the function was originally designed, and may potentially be insecure.
pub fn apply_round(
    state: &mut [BaseElement],
    op_code: BaseElement,
    op_value: BaseElement,
    step: usize,
) {
    let ark_idx = step % NUM_ROUNDS;

    // apply first half of Rescue round
    add_constants(state, ark_idx, 0);
    apply_sbox(state);
    apply_mds(state);

    // inject value into the state
    state[0] += op_code;
    state[1] += op_value;

    // apply second half of Rescue round
    add_constants(state, ark_idx, STATE_WIDTH);
    apply_inv_sbox(state);
    apply_mds(state);
}

pub fn add_constants(state: &mut [BaseElement], idx: usize, offset: usize) {
    for i in 0..STATE_WIDTH {
        state[i] += ARK[offset + i][idx];
    }
}

pub fn apply_sbox<E: FieldElement>(state: &mut [E]) {
    for element in state.iter_mut().take(STATE_WIDTH) {
        *element = element.exp(ALPHA.into());
    }
}

pub fn apply_inv_sbox(state: &mut [BaseElement]) {
    for element in state.iter_mut().take(STATE_WIDTH) {
        *element = element.exp(INV_ALPHA);
    }
}

pub fn apply_mds<E: FieldElement<BaseField = BaseElement>>(state: &mut [E]) {
    let mut result = [E::ZERO; STATE_WIDTH];
    let mut temp = [E::ZERO; STATE_WIDTH];
    for i in 0..STATE_WIDTH {
        for j in 0..STATE_WIDTH {
            temp[j] = E::from(MDS[i * STATE_WIDTH + j]) * state[j];
        }

        for &tmp_value in temp.iter() {
            result[i] += tmp_value;
        }
    }
    state.copy_from_slice(&result);
}

pub fn apply_inv_mds<E: FieldElement<BaseField = BaseElement>>(state: &mut [E]) {
    let mut result = [E::ZERO; STATE_WIDTH];
    let mut temp = [E::ZERO; STATE_WIDTH];
    for i in 0..STATE_WIDTH {
        for j in 0..STATE_WIDTH {
            temp[j] = E::from(INV_MDS[i * STATE_WIDTH + j]) * state[j];
        }

        for &tmp_value in temp.iter() {
            result[i] += tmp_value;
        }
    }
    state.copy_from_slice(&result);
}

// 128-BIT RESCUE CONSTANTS
// ================================================================================================
const ALPHA: u32 = 3;
const INV_ALPHA: u128 = 226854911280625642308916371969163307691;

const MDS: [BaseElement; STATE_WIDTH * STATE_WIDTH] = [
    BaseElement::new(315189521614069403867817270152032075784),
    BaseElement::new(10737242274749505456268020883296531251),
    BaseElement::new(164166492670388427786346110319108935134),
    BaseElement::new(282318813916891806489021925524031494414),
    BaseElement::new(339659984245804546554434478921876908973),
    BaseElement::new(97319381058524916656000376979320858814),
    BaseElement::new(141017807671871944240242749183803053011),
    BaseElement::new(271669633517564511702965675947905678154),
    BaseElement::new(330029911818464578106380298339390343164),
    BaseElement::new(37351365266361901988170671462637236976),
    BaseElement::new(260386862860725098262319886102680637202),
    BaseElement::new(161805458319902660573511017706877187889),
    BaseElement::new(220775330812333365987157544144955631442),
    BaseElement::new(172992909845374020745323861886602514703),
    BaseElement::new(8447293670850292346208742365584924315),
    BaseElement::new(276315004099450287580088164954743181255),
];

const INV_MDS: [BaseElement; STATE_WIDTH * STATE_WIDTH] = [
    BaseElement::new(212015899302823985314659753132599968692),
    BaseElement::new(222079945358547787481366483464725880498),
    BaseElement::new(313947036552775452548888741999726656951),
    BaseElement::new(94528877516599685906969597450601957552),
    BaseElement::new(201841258819571375352239737215387725848),
    BaseElement::new(42276963631701875238524357392500799145),
    BaseElement::new(332890116061360870847041499810748569092),
    BaseElement::new(3939991425276748394854935956419873430),
    BaseElement::new(239100689228321601709623770733501932352),
    BaseElement::new(178946314809288623489527367841505752988),
    BaseElement::new(270128331008291756180543638308504150653),
    BaseElement::new(315661002876081483102676309387501623498),
    BaseElement::new(298377528588644746682709801175581650901),
    BaseElement::new(114666605273067789843953739274063213369),
    BaseElement::new(279054651722812961169783459501878203576),
    BaseElement::new(308067640269163823896854342618197051588),
];

pub const ARK: [[BaseElement; NUM_ROUNDS]; STATE_WIDTH * 2] = [
    [
        BaseElement::new(73742662193393629993182617210984534396),
        BaseElement::new(53265540956785335308970867946461681393),
        BaseElement::new(14395595548581550072136442264588359269),
        BaseElement::new(122001776241989922016768881111033630021),
        BaseElement::new(60517382118002481956993039132628798754),
        BaseElement::new(242872884766759335785324964049644229294),
        BaseElement::new(4363347423120340347647334422662129280),
        BaseElement::new(36224510031696203479366212612960872957),
        BaseElement::new(48405253030503584410290697712994785780),
        BaseElement::new(81691558114273932307586556761543100315),
        BaseElement::new(315851285839738308287329276161693313425),
        BaseElement::new(326468515245013538774703881972225680443),
        BaseElement::new(43697512293048123577843997788308773455),
        BaseElement::new(311182552853825261047305944842224924215),
        BaseElement::new(23833044413239455428827669432473543240),
        BaseElement::new(7640791703119561504971867271087353186),
    ],
    [
        BaseElement::new(294241649061853322876594266104693176711),
        BaseElement::new(37163237225742447359704121711857363416),
        BaseElement::new(122453723578185362799857252115182955415),
        BaseElement::new(45955200056324872841369110391855073949),
        BaseElement::new(118224404177203231307646344308524770691),
        BaseElement::new(334905318181122708043147970432770442813),
        BaseElement::new(151456178618798089303785904835852898400),
        BaseElement::new(158324780313294970656577210958221752332),
        BaseElement::new(94987431711345870355583825474329298047),
        BaseElement::new(314293870425266938862923101612602484635),
        BaseElement::new(153975056764703018977481562856167540343),
        BaseElement::new(321383880935903155966493388921501530915),
        BaseElement::new(50057060110310193394516504439805601601),
        BaseElement::new(101740347373933108709122003348416870840),
        BaseElement::new(80845608757236703492016225128275757615),
        BaseElement::new(209519938465996994070512842713405349097),
    ],
    [
        BaseElement::new(158538539401072639862099558319550076686),
        BaseElement::new(221096166077280180974764042888991644280),
        BaseElement::new(58496669788416466040038464653643977917),
        BaseElement::new(59235259390239124162891762278360245334),
        BaseElement::new(337725857612570850944445340416668827103),
        BaseElement::new(232074846252364869196809445831737773796),
        BaseElement::new(50018412546799023168899671792323407156),
        BaseElement::new(166545598284411242433605578379265360252),
        BaseElement::new(41491163124497803255407972080635378902),
        BaseElement::new(302719082300742526890675313445319567341),
        BaseElement::new(193135973972933518870828237886863798021),
        BaseElement::new(230635877078223923040415038811686445073),
        BaseElement::new(138405289600908304802269329797084135857),
        BaseElement::new(185089342855265166563915858025522983409),
        BaseElement::new(43421407022492486112194101527865465264),
        BaseElement::new(62365388436267647533064120634464266870),
    ],
    [
        BaseElement::new(116358578177298194933445426886059838431),
        BaseElement::new(161426242690584918941198733450953748769),
        BaseElement::new(228752352631998151610775212885524543283),
        BaseElement::new(182846621472767704751329603405195985261),
        BaseElement::new(61911644581679112386499312030413349074),
        BaseElement::new(191090374127295994022314014407997806335),
        BaseElement::new(59079983632109588980783021622461415033),
        BaseElement::new(193859304217638223479173371326185841274),
        BaseElement::new(280938106646730498467301259432184740730),
        BaseElement::new(679464766810703810097965767355062873),
        BaseElement::new(150345637803188209699415557320545415720),
        BaseElement::new(139823638104054506965247243102295231737),
        BaseElement::new(53655583013674525883209345165753178194),
        BaseElement::new(126806292806004264446745284405742612689),
        BaseElement::new(9602891270757320013616862490986026227),
        BaseElement::new(160806286415414414379046661006476545066),
    ],
    [
        BaseElement::new(82429262549299942290847183493004485261),
        BaseElement::new(135862987622353414661673448620033990934),
        BaseElement::new(189653807408664613044858917026657980625),
        BaseElement::new(89333775516890774827962437297764936547),
        BaseElement::new(151495710594170316099539790651453416361),
        BaseElement::new(287288998844960276649854461883880913666),
        BaseElement::new(78065099645540746831460653583134588104),
        BaseElement::new(55063854082489962294956447144901184837),
        BaseElement::new(331958862978706756999748973740992156929),
        BaseElement::new(8168599451814692118441936734435571667),
        BaseElement::new(166002344081927954304873771936867289851),
        BaseElement::new(225556280578098393163620719229418290860),
        BaseElement::new(234470815157983004947611441850027217492),
        BaseElement::new(188323096432976273265052369652285099186),
        BaseElement::new(77086595049850596660690999278719011720),
        BaseElement::new(219177966622498447376602481443936826442),
    ],
    [
        BaseElement::new(153964862116746563988492365899737226989),
        BaseElement::new(171502007855719014010389694111716628578),
        BaseElement::new(69476260396969790693146402021744933499),
        BaseElement::new(154737674033120948700227987365296907637),
        BaseElement::new(321756280164272289841871040803703440350),
        BaseElement::new(131528659800906379821588177210148124019),
        BaseElement::new(80761083418432627134481526210881542477),
        BaseElement::new(250524460337537482950569449224813700391),
        BaseElement::new(230491494516843960542668710050516211970),
        BaseElement::new(45314269339319734203127091458645722622),
        BaseElement::new(2780762044206215421580528389917005833),
        BaseElement::new(165769058045453677221497711462583950139),
        BaseElement::new(259395388889719671782653113655841647262),
        BaseElement::new(219135320838134930923443959673366229256),
        BaseElement::new(286172465494565666647121151879345971089),
        BaseElement::new(147904845125332345734618546117273133070),
    ],
    [
        BaseElement::new(310538827479436149892724250590698914519),
        BaseElement::new(158907965876520949616863328303176330572),
        BaseElement::new(230609671293877243511889006223284127479),
        BaseElement::new(32424193637360906576442956294452323288),
        BaseElement::new(250107916917535224528378129994943394294),
        BaseElement::new(138628264101912804813977210615833233437),
        BaseElement::new(265168486075436613449458788630803272512),
        BaseElement::new(69216162599706897556278776240900218374),
        BaseElement::new(189445283838085809052254029811407633258),
        BaseElement::new(233141108584353453034002234415979233911),
        BaseElement::new(214406010671246827947835794343033790693),
        BaseElement::new(11153792801390339798262783617007369172),
        BaseElement::new(118114082982223329826045602989947510129),
        BaseElement::new(263157893448998999306850171729945394432),
        BaseElement::new(284751400376525550861233017183497639371),
        BaseElement::new(267697496976874759865163284761384997437),
    ],
    [
        BaseElement::new(93028746118909893246237533845189074002),
        BaseElement::new(185875552438512768131393810027777987752),
        BaseElement::new(185941035889835671403097747661105595079),
        BaseElement::new(253746202714926890236889780444552427226),
        BaseElement::new(101396399019525872501663112616210307683),
        BaseElement::new(215901816653704881294214215068798346060),
        BaseElement::new(201416315867789883891889554246377963162),
        BaseElement::new(251801358233276697762579413801911171612),
        BaseElement::new(192826288785653777157020265215517987662),
        BaseElement::new(15885268928012076989988458786521561031),
        BaseElement::new(3463161311202689884181131747640413605),
        BaseElement::new(79003969367131068546865741459306118251),
        BaseElement::new(69521903572951337445452502748565566496),
        BaseElement::new(301962999029915705994021697766389081208),
        BaseElement::new(9094956230559373758985913855137693312),
        BaseElement::new(144516981820451119929097195276243798745),
    ],
];
