use crate::model::{Entity, EntityColumns, EntityID};

// pub mod condition;

/// Wraps an entity with its ID, for example as a query result.
///
/// The wrapped value is accessible via `Deref`, so this should be mostly
/// transparent.
#[derive(Debug)]
pub struct WithID<T: Entity> {
    wrap: T,
    id: <T as Entity>::ID,
}

impl<T: Entity> WithID<T> {
    fn wrap(what: T, raw_id: i64) -> Self {
        Self {
            wrap: what,
            id: <T as Entity>::ID::from_raw_id(raw_id),
        }
    }
}

impl<T: Entity> WithID<T> {
    pub fn id(&self) -> <T as Entity>::ID {
        self.id
    }
}

impl<T: Entity> AsRef<T> for WithID<T> {
    fn as_ref(&self) -> &T {
        &self.wrap
    }
}

impl<T: Entity> std::ops::Deref for WithID<T> {
    type Target = T;
    fn deref(&self) -> &Self::Target {
        &self.wrap
    }
}

impl<T: Entity> std::ops::DerefMut for WithID<T> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.wrap
    }
}

type CacheIndex = (&'static str, std::any::TypeId, u64);

/// The query interface for a database.
///
/// As the query interface provides some level of caching, try to strive for as much sharing as
/// possible. Passing around `QueryInterface` references instead of `DB` references is a good way
/// to achieve this.
pub struct QueryInterface<'l> {
    db: &'l crate::DB,

    cache: std::sync::Mutex<std::collections::HashMap<CacheIndex, sqlite::Statement<'l>>>,
}

const NO_HASH: u64 = 0;

impl<'l> QueryInterface<'l> {
    pub fn new(db: &'l crate::DB) -> Self {
        Self {
            db,
            cache: std::sync::Mutex::new(std::collections::HashMap::new()),
        }
    }

    /// Helper function to process an expected single result
    /// Note that this errors out if there is more than a single result
    fn expect_one_result<T>(
        &self,
        stmt: &mut sqlite::Statement,
        with_result: &mut dyn FnMut(&mut sqlite::Statement) -> Option<T>,
    ) -> Option<T> {
        let state = stmt.next().ok()?;
        if state != sqlite::State::Row {
            return None;
        }

        let res = with_result(stmt);

        let state = stmt.next().ok()?;
        if state != sqlite::State::Done {
            return None;
        }

        res
    }

    fn cached_query<Return>(
        &self,
        context: &'static str,
        ty: std::any::TypeId,
        create: &dyn Fn() -> sqlite::Statement<'l>,
        with: &mut dyn FnMut(&mut sqlite::Statement<'l>) -> Return,
    ) -> Return {
        let mut cache = self.cache.lock().expect("Couldn't acquire cache?");
        let key = (context, ty, NO_HASH);
        let query = cache.entry(key).or_insert_with(create);

        query.reset().expect("Couldn't reset query");
        with(query)
    }

    fn cached_query_column<Column: crate::model::EntityColumns, Return>(
        &self,
        context: &'static str,
        ty: std::any::TypeId,
        variant: &Column,
        create: &dyn Fn() -> sqlite::Statement<'l>,
        with: &mut dyn FnMut(&mut sqlite::Statement<'l>) -> Return,
    ) -> Return {
        use std::hash::Hasher;

        let mut hasher = std::collections::hash_map::DefaultHasher::new();
        variant.hash(&mut hasher);
        let hash = hasher.finish();

        let mut cache = self.cache.lock().expect("Couldn't acquire cache?");
        let key = (context, ty, hash);
        let query = cache.entry(key).or_insert_with(create);

        query.reset().expect("Couldn't reset query");
        with(query)
    }

    /// Search for an entity by a property
    pub fn get_one_by<
        T: Entity<Column = C>,
        C: EntityColumns<Entity = T>,
        V: crate::model::Modelable,
    >(
        &self,
        c: C,
        val: V,
    ) -> Option<WithID<T>> {
        let table_name = <T as Entity>::table_name();
        let column_name = <T as Entity>::name(c.clone());

        self.cached_query_column(
            "get_one_by",
            std::any::TypeId::of::<T>(),
            &c,
            &|| {
                self.db
                    .conn
                    .prepare(&format!(
                        "SELECT * FROM \"{}\" WHERE \"{}\" = ?",
                        table_name, column_name
                    ))
                    .expect("")
            },
            &mut |stmt| {
                val.bind_to(stmt, 1).ok()?;

                self.expect_one_result(stmt, &mut |stmt| {
                    let id: i64 = stmt.read(0).ok()?;
                    let mut rd = crate::model::load::RowDeserializer::from_row(stmt);
                    Some(WithID::wrap(T::deserialize(&mut rd).ok()?, id))
                })
            },
        )
    }

    /// Search for an entity by ID
    pub fn get_one_by_id<I: EntityID<Entity = T>, T: Entity>(&self, id: I) -> Option<WithID<T>> {
        let table_name = <T as Entity>::table_name();

        self.cached_query(
            "get_one_by_id",
            std::any::TypeId::of::<T>(),
            &|| {
                self.db
                    .conn
                    .prepare(&format!("SELECT * FROM \"{}\" WHERE id = ?", table_name))
                    .expect("")
            },
            &mut |stmt| {
                id.bind_to(stmt, 1).ok()?;

                self.expect_one_result(stmt, &mut |stmt| {
                    let id: i64 = stmt.read(0).ok()?;
                    let mut rd = crate::model::load::RowDeserializer::from_row(stmt);
                    Some(WithID::wrap(T::deserialize(&mut rd).ok()?, id))
                })
            },
        )
    }

    /// Search for all entities matching a property
    pub fn get_all_by<
        T: Entity<Column = C>,
        C: EntityColumns<Entity = T>,
        V: crate::model::Modelable,
    >(
        &self,
        c: C,
        val: V,
    ) -> Option<Vec<WithID<T>>> {
        let table_name = <T as Entity>::table_name();
        let column_name = <T as Entity>::name(c.clone());

        self.cached_query_column(
            "get_all_by",
            std::any::TypeId::of::<T>(),
            &c,
            &|| {
                self.db
                    .conn
                    .prepare(&format!(
                        "SELECT * FROM \"{}\" WHERE {} = ?",
                        table_name, column_name
                    ))
                    .expect("")
            },
            &mut |stmt| {
                val.bind_to(stmt, 1).ok()?;

                let mut res = Vec::new();
                loop {
                    let state = stmt.next().ok()?;
                    if state == sqlite::State::Done {
                        break;
                    }

                    let id: i64 = stmt.read(0).ok()?;
                    let mut rd = crate::model::load::RowDeserializer::from_row(stmt);
                    res.push(WithID::wrap(T::deserialize(&mut rd).ok()?, id));
                }

                Some(res)
            },
        )
    }

    /// Add an entity to its table
    pub fn add<T: Entity + serde::Serialize>(&self, m: &T) -> Option<<T as Entity>::ID> {
        self.cached_query(
            "get_all_by",
            std::any::TypeId::of::<T>(),
            &|| {
                let placeholders = (0..(<T as Entity>::column_count() - 1))
                    .map(|_| "?".to_string())
                    .collect::<Vec<_>>()
                    .join(",");

                self.db
                    .conn
                    .prepare(&format!(
                        "INSERT INTO \"{}\" VALUES (NULL, {}) RETURNING \"id\"",
                        <T as Entity>::table_name(),
                        placeholders
                    ))
                    .expect("")
            },
            &mut |stmt| {
                crate::model::store::serialize_into(stmt, m).ok()?;

                let rowid = self.expect_one_result(stmt, &mut |stmt| stmt.read::<i64>(0).ok())?;

                Some(<T as Entity>::ID::from_raw_id(rowid))
            },
        )
    }
}
