use super::Modelable;
use sqlite::Bindable;

macro_rules! integral {
    ($i:ty) => {
        impl Modelable for $i {
            fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
                (*self as i64).bind(stmt, col)
            }
            fn build_from(
                stmt: &sqlite::Statement,
                col_offset: usize,
            ) -> sqlite::Result<(Self, usize)>
            where
                Self: Sized,
            {
                stmt.read::<i64>(col_offset).map(|x| (x as Self, 1))
            }
        }
    };
}

integral!(i8);
integral!(u8);
integral!(i16);
integral!(u16);
integral!(i32);
integral!(u32);
integral!(i64);
integral!(u64);

impl Modelable for f64 {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        self.bind(stmt, col)
    }
    fn build_from(stmt: &sqlite::Statement, col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        stmt.read(col_offset).map(|x| (x, 1))
    }
}

impl Modelable for bool {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        let val = if self == &true { 1i64 } else { 0i64 };
        val.bind(stmt, col)
    }
    fn build_from(_stmt: &sqlite::Statement, _col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        unreachable!("sqlite only gives Strings back, not &strs!");
    }
}

impl<'a> Modelable for &'a str {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        self.bind(stmt, col)
    }
    fn build_from(_stmt: &sqlite::Statement, _col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        unreachable!("sqlite only gives Strings back, not &strs!");
    }
}

impl Modelable for std::string::String {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        self.as_str().bind(stmt, col)
    }
    fn build_from(stmt: &sqlite::Statement, col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        stmt.read(col_offset).map(|x| (x, 1))
    }
}

impl<'a> Modelable for &'a [u8] {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        self.bind(stmt, col)
    }
    fn build_from(_stmt: &sqlite::Statement, _col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        unreachable!("sqlite only gives Vec<u8> back, not &[u8]!");
    }
}

impl Modelable for Vec<u8> {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()> {
        self.bind(stmt, col)
    }
    fn build_from(stmt: &sqlite::Statement, col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized,
    {
        stmt.read(col_offset).map(|x| (x, 1))
    }
}
