use super::Modelable;
use serde::de::Visitor;

pub struct RowDeserializer<'de, 'l> {
    row: &'de sqlite::Statement<'l>,
    col_index: usize,
}

impl<'de, 'l> RowDeserializer<'de, 'l> {
    pub fn from_row(row: &'de sqlite::Statement<'l>) -> Self {
        // we skip the rowid by starting at index 1
        Self { row, col_index: 1 }
    }
}

impl<'de, 'a, 'l> serde::de::Deserializer<'de> for &'a mut RowDeserializer<'de, 'l> {
    type Error = super::ModelError;

    fn deserialize_any<V: Visitor<'de>>(self, _v: V) -> Result<V::Value, Self::Error> {
        todo!()
    }

    fn deserialize_bool<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_bool(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_i8<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_i8(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_i16<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_i16(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_i32<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_i32(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_i64<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_i64(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_string<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_string(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_byte_buf<V: Visitor<'de>>(self, v: V) -> Result<V::Value, Self::Error> {
        let built = Modelable::build_from(self.row, self.col_index)
            .map_err(|e| Self::Error::LoadError(e.to_string()))?;
        let res = v.visit_byte_buf(built.0);
        self.col_index += built.1;
        res
    }

    fn deserialize_struct<V: Visitor<'de>>(
        self,
        _name: &'static str,
        _fields: &'static [&'static str],
        v: V,
    ) -> Result<V::Value, Self::Error> {
        v.visit_seq(self)
    }

    fn deserialize_newtype_struct<V: Visitor<'de>>(
        self,
        _name: &'static str,
        v: V,
    ) -> Result<V::Value, Self::Error> {
        v.visit_seq(self)
    }

    serde::forward_to_deserialize_any! {
        i128 u8 u16 u32 u64 u128 f32 f64 char str
        bytes option unit unit_struct seq tuple
        tuple_struct map enum identifier ignored_any
    }
}

impl<'de, 'l> serde::de::SeqAccess<'de> for RowDeserializer<'de, 'l> {
    type Error = super::ModelError;

    fn next_element_seed<T: serde::de::DeserializeSeed<'de>>(
        &mut self,
        seed: T,
    ) -> Result<Option<T::Value>, Self::Error> {
        seed.deserialize(self).map(Some)
    }
}
