mod create;
pub(crate) mod load;
pub(crate) mod store;

// Modelable implementations
mod modelable;

#[derive(Debug)]
pub enum ModelError {
    DBError(sqlite::Error),
    LoadError(String),
    StoreError(String),
    EmptyStoreError,
    CreateError,
}

impl From<sqlite::Error> for ModelError {
    fn from(e: sqlite::Error) -> Self {
        Self::DBError(e)
    }
}

impl From<ModelError> for sqlite::Error {
    fn from(e: ModelError) -> Self {
        match e {
            ModelError::DBError(e) => e,
            _ => panic!(),
        }
    }
}

impl std::fmt::Display for ModelError {
    fn fmt(&self, fmt: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        fmt.write_fmt(format_args!("{:?}", self))
    }
}

impl serde::ser::Error for ModelError {
    fn custom<T: std::fmt::Display>(msg: T) -> Self {
        Self::StoreError(format!("{}", msg))
    }
}

impl serde::de::Error for ModelError {
    fn custom<T: std::fmt::Display>(msg: T) -> Self {
        Self::LoadError(format!("{}", msg))
    }
}

impl std::error::Error for ModelError {}

/// A database value, aka a single column of a single row
pub trait Modelable {
    fn bind_to(&self, stmt: &mut sqlite::Statement, col: usize) -> sqlite::Result<()>;
    fn build_from(stmt: &sqlite::Statement, col_offset: usize) -> sqlite::Result<(Self, usize)>
    where
        Self: Sized;
}

/// A database entity, aka a struct representing a row in a table
pub trait Entity: 'static + for<'de> serde::Deserialize<'de> + serde::Serialize {
    type Column: EntityColumns + 'static + Copy;
    type ID: EntityID;
    fn table_name() -> &'static str;
    fn column_count() -> usize
    where
        Self: Sized;
    fn index(c: Self::Column) -> usize
    where
        Self: Sized;
    fn name(c: Self::Column) -> &'static str
    where
        Self: Sized;
    fn values(&self) -> Vec<&dyn Modelable>;

    fn foreign_keys() -> &'static [&'static dyn EntityForeignKey<Self::Column>];
}

/// Trait representing the columns of a database entity
pub trait EntityColumns: 'static + PartialEq + From<usize> + std::hash::Hash + Clone {
    type Entity: Entity;
}

/// Trait for entity IDs in the database
pub trait EntityID: 'static + std::fmt::Debug + Copy + Modelable {
    type Entity: Entity;
    fn from_raw_id(raw: i64) -> Self;
    fn raw_id(&self) -> i64;
}

/// Trait for a foreign key relationship
pub trait EntityForeignKey<T: EntityColumns> {
    fn local_column(&self) -> &T;
    fn foreign_table_name(&self) -> &'static str;
    fn foreign_column_name(&self) -> &'static str;
}

/// Trait for an index over a column
pub trait Index {
    type IndexedEntity : Entity;

    fn index_name() -> &'static str
    where
        Self: Sized;
    fn columns() -> &'static [<<Self as Index>::IndexedEntity as Entity>::Column]
    where
        Self: Sized;
    fn unique() -> bool
    where
        Self: Sized;
}

/// How we describe an entire schema
#[derive(Debug)]
pub struct SchemaModel {
    drop: Vec<String>,
    create: Vec<String>,
}

impl SchemaModel {
    pub fn new() -> Self {
        Self {
            drop: Vec::new(),
            create: Vec::new(),
        }
    }

    pub fn entity<E: Entity>(mut self) -> Self {
        let (drop, create) = create::sql_for_table::<E>();
        self.drop.push(drop);
        self.create.push(create);
        self
    }

    pub fn index<I: Index>(
        mut self,
    ) -> Self {
        let (drop, create) = create::sql_for_index::<I>();
        self.drop.push(drop);
        self.create.push(create);
        self
    }

    pub fn add<E: Entity>(mut self) -> Self {
        let (drop, create) = create::sql_for_table::<E>();
        self.drop.push(drop);
        self.create.push(create);
        self
    }

    pub fn drop(&self) -> &Vec<String> {
        &self.drop
    }
    pub fn create(&self) -> &Vec<String> {
        &self.create
    }
}

impl Default for SchemaModel {
    fn default() -> Self {
        Self::new()
    }
}
