//! `microrm` is a crate providing a lightweight ORM on top of SQLite.
//!
//! Unlike fancier ORM systems, microrm is intended to be extremely lightweight
//! and code-light, which means that by necessity it is opinionated, and thus
//! lacks the power and flexibility of, say, SeaORM or Diesel. In particular,
//! `microrm` currently makes no attempts to provide database migration support.
//!
//! `microrm` provides two components: modeling and querying. The intention is
//! that the modelling is built statically; dynamic models are not directly
//! supported though are possible. However, since by design microrm does not
//! touch database contents for tables not defined in its model, using raw SQL
//! for any needed dynamic components may be a better choice.
//!
//! Querying supports a small subset of SQL expressed as type composition.
//!
//! A simple example using an SQLite table as an (indexed) key/value store
//! might look something like this:
//! ```rust
//! use microrm::{Entity,make_index};
//! #[derive(Debug,Entity,serde::Serialize,serde::Deserialize)]
//! pub struct KVStore {
//!     pub key: String,
//!     pub value: String
//! }
//!
//! // the !KVStoreIndex here means a type representing a unique index named KVStoreIndex
//! make_index!(!KVStoreIndex, KVStoreColumns::Key);
//!
//! let schema = microrm::model::SchemaModel::new()
//!     .add::<KVStore>()
//!     .index::<KVStoreIndex>();
//!
//! // dump the schema in case you want to inspect it manually
//! for create_sql in schema.create() {
//!     println!("{};", create_sql);
//! }
//!
//! let db = microrm::DB::new_in_memory(schema).unwrap();
//! let qi = db.query_interface();
//!
//! qi.add(&KVStore {
//!     key: "a_key".to_string(),
//!     value: "a_value".to_string()
//! });
//!
//! // because KVStoreIndex indexes key, this is a logarithmic lookup
//! let qr = qi.get_one_by(KVStoreColumns::Key, "a_key");
//!
//! assert_eq!(qr.is_some(), true);
//! assert_eq!(qr.as_ref().unwrap().key, "a_key");
//! assert_eq!(qr.as_ref().unwrap().value, "a_value");
//! ```
//!
//! The schema output from the loop is (details subject to change based on internals):
//! ```sql
//! CREATE TABLE IF NOT EXISTS "kv_store" (id integer primary key,"key" text,"value" text);
//! CREATE UNIQUE INDEX "kv_store_index" ON "kv_store" ("key");
//! ```


mod meta;
pub mod model;
pub mod query;

use meta::Metaschema;
use model::Entity;

pub use microrm_macros::{Entity, Modelable, make_index};
pub use query::{QueryInterface, WithID};

// no need to show the re-exports in the documentation
#[doc(hidden)]
pub mod re_export {
    pub use serde;
    pub use serde_json;
    pub use sqlite;
}

#[derive(Debug)]
pub enum DBError {
    ConnectFailure,
    EarlyFailure(sqlite::Error),
    NoSchema,
    DifferentSchema,
    DropFailure,
    CreateFailure,
    SanityCheckFailure,
}

#[derive(PartialEq, Debug)]
pub enum CreateMode {
    /// The database must exist and have a valid schema already
    MustExist,
    /// It's fine if the database doesn't exist, but it must have a valid schema if it does
    AllowNewDatabase,
    /// Nuke the contents if need be, just get the database
    AllowSchemaUpdate,
}

impl std::fmt::Display for DBError {
    fn fmt(&self, fmt: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        fmt.write_fmt(format_args!("Database error: {:?}", self))
    }
}

impl std::error::Error for DBError {}

/// SQLite database connection
pub struct DB {
    conn: sqlite::Connection,
    schema_hash: String,
    schema: model::SchemaModel,
}

impl DB {
    pub fn new(schema: model::SchemaModel, path: &str, mode: CreateMode) -> Result<Self, DBError> {
        Self::from_connection(
            sqlite::Connection::open(path).map_err(|_| DBError::ConnectFailure)?,
            schema,
            mode,
        )
    }

    /// Mostly for use in tests, but may be useful in some applications as well.
    pub fn new_in_memory(schema: model::SchemaModel) -> Result<Self, DBError> {
        Self::from_connection(
            sqlite::Connection::open(":memory:").map_err(|_| DBError::ConnectFailure)?,
            schema,
            CreateMode::AllowNewDatabase,
        )
    }

    /// Get a query interface for this DB connection
    pub fn query_interface(&self) -> query::QueryInterface {
        query::QueryInterface::new(self)
    }

    pub fn recreate_schema(&self) -> Result<(), DBError> {
        self.create_schema()
    }

    fn from_connection(
        conn: sqlite::Connection,
        schema: model::SchemaModel,
        mode: CreateMode,
    ) -> Result<Self, DBError> {
        let sig = Self::calculate_schema_hash(&schema);
        let ret = Self {
            conn,
            schema_hash: sig,
            schema: schema.add::<meta::Metaschema>(),
        };
        ret.check_schema(mode)?;
        Ok(ret)
    }

    fn calculate_schema_hash(schema: &model::SchemaModel) -> String {
        use sha2::Digest;

        let mut hasher = sha2::Sha256::new();
        schema
            .drop()
            .iter()
            .map(|sql| hasher.update(sql.as_bytes()))
            .count();
        schema
            .create()
            .iter()
            .map(|sql| hasher.update(sql.as_bytes()))
            .count();

        base64::encode(hasher.finalize())
    }

    fn check_schema(&self, mode: CreateMode) -> Result<(), DBError> {
        let mut has_metaschema = false;
        self.conn
            .iterate(
                format!(
                    "SELECT * FROM \"sqlite_master\" WHERE \"type\"='table' AND \"name\"='{}'",
                    Metaschema::table_name()
                ),
                |row| {
                    println!("row: {:?}", row);
                    has_metaschema = true;
                    true
                },
            )
            .map_err(DBError::EarlyFailure)?;

        if mode != CreateMode::MustExist {
            println!("Creating schema!");
            return self.create_schema();
        }

        let qi = query::QueryInterface::new(self);
        let hash = qi.get_one_by(meta::MetaschemaColumns::Key, "schema_hash");

        if hash.is_none() {
            if mode == CreateMode::MustExist {
                return Err(DBError::NoSchema);
            }
            return self.create_schema();
        } else if hash.unwrap().value != self.schema_hash {
            if mode != CreateMode::AllowSchemaUpdate {
                return Err(DBError::DifferentSchema);
            }
            self.drop_schema()?;
            return self.create_schema();
        }

        Ok(())
    }

    fn drop_schema(&self) -> Result<(), DBError> {
        for ds in self.schema.drop() {
            self.conn.execute(ds).map_err(|_| DBError::DropFailure)?;
        }
        Ok(())
    }

    fn create_schema(&self) -> Result<(), DBError> {
        for cs in self.schema.create() {
            self.conn.execute(cs).map_err(|_| DBError::CreateFailure)?;
        }

        let qi = query::QueryInterface::new(self);

        let add_result = qi.add(&meta::Metaschema {
            key: "schema_hash".to_string(),
            value: self.schema_hash.clone(),
        });

        assert!(add_result.is_some());

        let sanity_check = qi.get_one_by(meta::MetaschemaColumns::Key, "schema_hash");
        assert!(sanity_check.is_some());
        assert_eq!(sanity_check.unwrap().value, self.schema_hash);

        Ok(())
    }
}

#[cfg(test)]
mod test {
    use super::DB;

    #[derive(serde::Serialize, serde::Deserialize, crate::Entity)]
    #[microrm_internal]
    pub struct S1 {
        an_id: i32,
    }

    fn simple_schema() -> super::model::SchemaModel {
        super::model::SchemaModel::new().add::<S1>()
    }

    #[test]
    fn in_memory_schema() {
        let _db = DB::new_in_memory(simple_schema());
        drop(_db);
    }

    #[derive(serde::Serialize, serde::Deserialize, crate::Entity)]
    #[microrm_internal]
    pub struct S2 {
        #[microrm_foreign]
        parent_id: S1ID,
    }

    #[test]
    fn simple_foreign_key() {
        let db = DB::new_in_memory(super::model::SchemaModel::new().add::<S1>().add::<S2>())
            .expect("Can't connect to in-memory DB");
        let qi = db.query_interface();

        let id = qi.add(&S1 { an_id: -1 }).expect("Can't add S1");
        let child_id = qi.add(&S2 { parent_id: id }).expect("Can't add S2");

        qi.get_one_by_id(child_id).expect("Can't get S2 instance");
    }

    microrm_macros::make_index_internal!(S2ParentIndex, S2Columns::ParentId);
}

#[cfg(test)]
mod test2 {
    use crate::{Entity,make_index};
    #[derive(Debug,Entity,serde::Serialize,serde::Deserialize)]
    #[microrm_internal]
    pub struct KVStore {
        pub key: String,
        pub value: String
    }

    // the !KVStoreIndex here means a type representing a unique index named KVStoreIndex
    microrm_macros::make_index_internal!(!KVStoreIndex, KVStoreColumns::Key);

    #[test]
    fn dump_test() {
        let schema = crate::model::SchemaModel::new()
            .add::<KVStore>()
            .index::<KVStoreIndex>();

        // dump the schema in case you want to inspect it manually
        for create_sql in schema.create() {
            println!("{};", create_sql);
        }

        let db = crate::DB::new_in_memory(schema).unwrap();
        let qi = db.query_interface();

        qi.add(&KVStore {
            key: "a_key".to_string(),
            value: "a_value".to_string()
        });

        // because KVStoreIndex indexes key, this is a logarithmic lookup
        let qr = qi.get_one_by(KVStoreColumns::Key, "a_key");

        assert_eq!(qr.is_some(), true);
        assert_eq!(qr.as_ref().unwrap().key, "a_key");
        assert_eq!(qr.as_ref().unwrap().value, "a_value");
    }
}
