//! Validation passes for MIR.
//!
//! This module provides a collection of functions that check properties
//! that should be ensured by the structure of the compiler, but which we might
//! mess up by mistake if we're not careful.

use super::{MirEdge, MirGraph, MirNode};

/// Check that [`FunctionDependency`](MirEdge::FunctionDependency) edges are used
/// for conveying functions, and nothing else, and that functions are *only* conveyed
/// using the correct edge kind.
fn only_function_deps_for_functions(graph: &MirGraph) -> bool {
    true
}

/// This pass ensures that the callee for each of the function calls in a program
/// is able to be known statically.
///
/// The requirements are these:
/// - Functions are created by `FunctionDefinition` and `PartialApply` nodes
/// - `FunctionDependency` edges convey only functions
/// - Only `FunctionDependency` edges convey functions
/// - Functions cannot be returned by `Decision` nodes
/// - Functions can be returned by `Loop` and `RecursiveEnvironment` nodes
fn statically_resolvable_functions(graph: &MirGraph) -> bool {
    true
}

/// Check that all node arguments match with the expected types.
fn arg_types(graph: &MirGraph) -> bool {
    true
}

pub(super) fn all(graph: &MirGraph) -> Result<(), Vec<&'static str>> {
    let mut failures = Vec::new();
    macro_rules! checks {
        ($($name:ident),* $(,)?) => {
            $(
                if ! $name(graph) {
                    failures.push(stringify!($name));
                }
            )*
        }
    }
    checks! {
        arg_types,
        statically_resolvable_functions,
    }
    if failures.is_empty() {
        Ok(())
    } else {
        Err(failures)
    }
}
