use handlebars::{Context, Handlebars, Helper, HelperResult, Output, RenderContext};
use serde_json::json;

use crate::model::Field;

pub fn json_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value();
    out.write(&serde_json::to_string_pretty(param).unwrap())?;
    Ok(())
}

pub fn title_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value().as_str().unwrap();
    let mut chars = param.chars();

    let result = match chars.next() {
        None => String::new(),
        Some(c) => c.to_uppercase().collect::<String>() + chars.as_str(),
    };

    out.write(&result)?;
    Ok(())
}

pub fn form_field_helper(
    h: &Helper,
    hb: &Handlebars,
    _ctx: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let field = serde_json::from_value::<Field>(h.param(0).unwrap().value().to_owned())?;
    let initial_value = h.param(1).unwrap().value();

    let field_template = format!("form/{}_field", json!(field.content_type).as_str().unwrap());

    match hb.render(
        &field_template,
        &json!({"field": field, "initial_value": initial_value}),
    ) {
        Ok(field) => out.write(&field)?,
        Err(err) => out.write(&format!("<span class=\"error\">{}</span>", err.desc))?,
    }

    Ok(())
}

pub fn field_value_helper(
    h: &Helper,
    _hb: &Handlebars,
    _ctx: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let field = h.param(0).unwrap().value();
    if field.is_string() {
        out.write(field.as_str().unwrap())?;
    } else if field.is_object() {
        if let Some(model) = field.get("$ref") {
            let ref_id = field.get("$id").unwrap().as_str().unwrap();
            out.write(format!("{} ({})", model.as_str().unwrap(), ref_id).as_str())?
        } else if let Some(id) = field.get("$oid") {
            out.write(id.as_str().unwrap())?
        }
    }
    Ok(())
}

pub fn title_field_helper(
    h: &Helper,
    _hb: &Handlebars,
    _ctx: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let field = h.param(0).unwrap().value();

    if let Some(title) = field.get("title") {
        out.write(title.as_str().unwrap())?;
    } else if let Some(title) = field.get("name") {
        out.write(title.as_str().unwrap())?;
    } else {
        let (_, title) = field
            .as_object()
            .unwrap()
            .iter()
            .find(|(_, value)| value.is_string())
            .unwrap();

        out.write(title.as_str().unwrap())?
    }

    Ok(())
}
