use serde::{Deserialize, Serialize};
use std::fmt::Display;

#[derive(Clone, Debug, PartialEq)]
pub struct Version(pub u8, pub u8, pub u8);

impl Display for Version {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(format!("{}.{}.{}", self.0, self.1, self.2).as_str())
    }
}

impl Serialize for Version {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(format!("{}", self).as_str())
    }
}

impl<'de> Deserialize<'de> for Version {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let s: &str = Deserialize::deserialize(deserializer)?;

        let version: Version = s
            .split(".")
            .map(|part| part.parse::<u8>().unwrap())
            .collect();

        Ok(version)
    }
}

impl FromIterator<u8> for Version {
    fn from_iter<T: IntoIterator<Item = u8>>(iter: T) -> Self {
        let mut nums = iter.into_iter();

        let v1 = nums.next().unwrap();
        let v2 = nums.next().unwrap();
        let v3 = nums.next().unwrap();

        Self(v1, v2, v3)
    }
}
