use anyhow::{Error, Result};
use serde::{Deserialize, Serialize};
use std::fmt::Debug;
use std::fs;
use std::path::Path;

#[derive(Clone, Debug, Serialize, Deserialize)]
#[serde(rename_all = "snake_case")]
pub enum ContentType {
    Text,
    Image,
    Reference,
    List,
}

#[derive(Clone, Debug, Deserialize, Serialize)]
pub struct Model {
    pub name: String,
    pub fields: Vec<Field>,
}

impl Model {
    pub fn new(name: &str) -> Self {
        Self {
            name: String::from(name),
            fields: vec![],
        }
    }

    pub fn from_file<P>(path: P) -> Result<Self>
    where
        P: Debug + AsRef<Path>,
    {
        if let Ok(raw) = fs::read_to_string(path) {
            return Ok(serde_json::from_str(&raw.as_str())?);
        }
        Err(Error::msg("Failed to parse model."))
    }

    pub fn add_field(&mut self, field: Field) {
        self.fields.push(field)
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct Field {
    pub name: String,
    #[serde(rename = "type")]
    pub content_type: ContentType,
    pub of: Option<String>,
}

impl Field {
    pub fn text(name: &str) -> Self {
        Self {
            name: String::from(name),
            content_type: ContentType::Text,
            of: None,
        }
    }

    pub fn image(name: &str) -> Self {
        Self {
            name: String::from(name),
            content_type: ContentType::Image,
            of: None,
        }
    }

    pub fn reference(name: &str, of: &Model) -> Self {
        Self {
            name: String::from(name),
            content_type: ContentType::Reference,
            of: Some(of.name.clone()),
        }
    }
}
