pub mod version;

use anyhow::Result;
use serde::{Deserialize, Serialize};
use serde_json::json;
use std::env;
use std::fs;
use version::Version;

use crate::embedded::{Compiled, ExternalScripts};
use crate::model::Model;

#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct Project {
    pub name: String,
    pub version: Version,
    #[serde(skip)]
    pub db_uri: String,
    #[serde(skip)]
    pub models: Vec<Model>,
}

impl Project {
    pub fn default(name: &str) -> Self {
        Project {
            name: String::from(name),
            version: Version(0, 1, 0),
            models: vec![],
            db_uri: String::new(),
        }
    }

    pub fn from_dir(dir: &str) -> Result<Self> {
        let db_uri = env::var("DATABASE_URI")?;

        let models_path = format!("{}/models/", dir);
        let config_path = format!("{}/mezzo.json", dir);

        let raw_config = fs::read_to_string(config_path)?;
        let project = serde_json::from_str::<Self>(&raw_config.as_str())?;

        let mut models = vec![];
        let models_dir = fs::read_dir(models_path)?;

        for res in models_dir {
            if let Ok(entry) = res {
                let model = Model::from_file(entry.path())?;
                models.push(model);
            }
        }

        Ok(Self {
            models,
            db_uri,
            ..project
        })
    }

    pub fn set_db_uri(&mut self, uri: String) {
        self.db_uri = uri;
    }

    pub fn generate_template(&self) -> Result<()> {
        // Create project directories
        fs::create_dir(&self.name)?;

        for folder in vec![
            "models",
            ".mezzo",
            ".mezzo/public",
            ".mezzo/public/scripts",
            ".mezzo/public/scripts/external",
            ".mezzo/public/styles",
        ] {
            fs::create_dir(format!("{}/{}", self.name, folder))?;
        }

        // Create model files
        for model in &self.models {
            let path = format!("{}/models/{}.json", self.name, model.name);
            fs::write(&path, &json!(model).to_string())?;
        }

        // Copy static files
        for path in ExternalScripts::iter() {
            let file = ExternalScripts::get(&path).unwrap();
            let content = std::str::from_utf8(&file.data)?;
            fs::write(
                format!("{}/.mezzo/public/scripts/external/{}", self.name, &path),
                content,
            )?;
        }

        // Copy compiled files
        for path in Compiled::iter() {
            let file = Compiled::get(&path).unwrap();
            let content = std::str::from_utf8(&file.data)?;
            fs::write(format!("{}/.mezzo/public/{}", self.name, &path), content)?;
        }

        // Misc
        fs::write(
            format!("{}/.env", self.name),
            format!("DATABASE_URI={}", self.db_uri),
        )?;

        fs::write(format!("{}/.gitignore", self.name), ".env")?;

        fs::write(
            format!("{}/mezzo.json", self.name),
            serde_json::to_string(self).unwrap(),
        )?;

        Ok(())
    }
}
