use anyhow::Result;
use futures::TryStreamExt;
use mongodb::{
    bson::{doc, oid::ObjectId, Document},
    options::UpdateOptions,
    Database,
};
use serde::{Deserialize, Serialize};
use serde_json::json;

#[derive(Serialize, Deserialize)]
pub enum ApiResult {
    #[serde(rename = "message")]
    Ok(String),
    #[serde(rename = "data")]
    Data(serde_json::Value),
    #[serde(rename = "error")]
    Err(String),
}

#[derive(Clone)]
pub struct Client {
    db: Database,
}

impl Client {
    pub fn new(db: Database) -> Self {
        Self { db }
    }

    pub async fn find_all(&self, collection: &String) -> Result<ApiResult> {
        let mut cursor = self
            .db
            .collection::<serde_json::Value>(collection.as_str())
            .find(None, None)
            .await?;

        let mut data = Vec::new();

        while let Some(value) = cursor.try_next().await? {
            data.push(value);
        }

        Ok(ApiResult::Data(json!(data)))
    }

    pub async fn find_by_id(&self, collection: &String, id: &String) -> Result<ApiResult> {
        let oid = ObjectId::parse_str(id)?;

        let data = self
            .db
            .collection::<serde_json::Value>(collection.as_str())
            .find_one(doc! {"_id": oid}, None)
            .await?;

        if let Some(entity) = data {
            Ok(ApiResult::Data(json!(entity)))
        } else {
            Ok(ApiResult::Err(String::from("Entity was not found.")))
        }
    }

    pub async fn create(&self, collection: &String, value: Document) -> Result<ApiResult> {
        self.db
            .collection(collection.as_str())
            .insert_one(value, None)
            .await?;

        Ok(ApiResult::Ok(format!("Created.")))
    }

    pub async fn update(
        &self,
        collection: &String,
        id: &String,
        value: Document,
    ) -> Result<ApiResult> {
        let oid = ObjectId::parse_str(id)?;

        let result = self
            .db
            .collection::<serde_json::Value>(collection.as_str())
            .update_one(
                doc! { "_id": oid },
                doc! { "$set": value },
                UpdateOptions::default(),
            )
            .await?;

        Ok(ApiResult::Data(json!({ "updated": result.modified_count })))
    }

    pub async fn remove(&self, collection: &String, id: &String) -> Result<ApiResult> {
        let oid = ObjectId::parse_str(id)?;

        let result = self
            .db
            .collection::<serde_json::Value>(collection.as_str())
            .delete_one(doc! {"_id": oid}, None)
            .await?;

        Ok(ApiResult::Data(json!({ "deleted": result.deleted_count } )))
    }
}
