use anyhow::Result;
use std::collections::HashMap;

use crate::args::arg_or_prompt;
use crate::model::{Field, Model};
use crate::project::Project;
use crate::server::start_server;
use crate::utils::io::prompt;

pub trait Command {
    fn execute(&self) -> Result<()>;
}

pub struct NewProjectCommand {
    name: String,
}

impl NewProjectCommand {
    pub fn new(arg: Option<&String>) -> Self {
        let name = arg_or_prompt(arg, "Give your project a name: ");
        Self { name }
    }
}

impl Command for NewProjectCommand {
    fn execute(&self) -> Result<()> {
        let mut project = Project::default(&self.name);

        let mut author_model = Model::new("author");
        author_model.add_field(Field::text("name"));

        let mut post_model = Model::new("post");
        post_model.add_field(Field::text("title"));
        post_model.add_field(Field::text("date"));
        post_model.add_field(Field::image("cover"));
        post_model.add_field(Field::reference("author", &author_model));

        let db_uri = prompt(
            "I am kindly asking for you mongo db uri: ",
            "mongodb://localhost:27017/mezzo",
        );

        project.set_db_uri(db_uri);

        project.models.push(post_model);
        project.models.push(author_model);

        project.generate_template()?;

        println!("\n✨ {} is ready!", project.name);
        Ok(())
    }
}

pub struct StartProjectCommand {
    root: String,
}

impl StartProjectCommand {
    pub fn new(arg: Option<&String>) -> Self {
        let default = String::from(".");
        let root = arg.unwrap_or(&default).to_owned();

        Self { root }
    }
}

impl Command for StartProjectCommand {
    fn execute(&self) -> Result<()> {
        start_server(self.root.clone())?;
        Ok(())
    }
}

pub struct HelpCommand {}

impl Command for HelpCommand {
    fn execute(&self) -> Result<()> {
        let mut commands = HashMap::<&str, String>::new();

        commands.insert(
            "new [project name]",
            String::from("Creates an empty project."),
        );
        commands.insert(
            "start [root directory]",
            String::from("Starts the mezzo app."),
        );
        commands.insert("help", String::from("Prints this message."));

        println!("Usage: mezzo [command] [args] [flags]\n\nCommands");

        for (command, help_text) in commands {
            println!("    {}:\t{}", command, help_text);
        }

        Ok(())
    }
}
