use anyhow::Result;
use serde::{Deserialize, Serialize};
use std::env;
use std::fmt::Display;
use std::fs;

use crate::embedded::Compiled;
use crate::model::Model;

#[derive(Clone, Debug, PartialEq)]
pub struct Version(u8, u8, u8);

impl Display for Version {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(format!("{}.{}.{}", self.0, self.1, self.2).as_str())
    }
}

impl Serialize for Version {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(format!("{}", self).as_str())
    }
}

impl<'de> Deserialize<'de> for Version {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let s: &str = Deserialize::deserialize(deserializer)?;

        let version: Version = s
            .split(".")
            .map(|part| part.parse::<u8>().unwrap())
            .collect();

        Ok(version)
    }

    fn deserialize_in_place<D>(deserializer: D, place: &mut Self) -> Result<(), D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        // Default implementation just delegates to `deserialize` impl.
        *place = Deserialize::deserialize(deserializer)?;
        Ok(())
    }
}

impl FromIterator<u8> for Version {
    fn from_iter<T: IntoIterator<Item = u8>>(iter: T) -> Self {
        let mut nums = iter.into_iter();

        let v1 = nums.next().unwrap();
        let v2 = nums.next().unwrap();
        let v3 = nums.next().unwrap();

        Self(v1, v2, v3)
    }
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct Project {
    pub name: String,
    pub version: Version,

    #[serde(skip)]
    pub db_uri: String,

    #[serde(skip)]
    pub models: Vec<Model>,
}

impl Project {
    pub fn default(name: &str) -> Self {
        Project {
            name: String::from(name),
            version: Version(0, 1, 0),
            models: vec![],
            db_uri: String::new(),
        }
    }

    pub fn from_dir(dir: &str) -> Result<Self> {
        let db_uri = env::var("DATABASE_URI")?;

        let models_path = format!("{}/models/", dir);
        let config_path = format!("{}/mezzo.json", dir);

        let raw_config = fs::read_to_string(config_path)?;
        let project = serde_json::from_str::<Self>(&raw_config.as_str())?;

        let mut models = vec![];
        let models_dir = fs::read_dir(models_path)?;

        for res in models_dir {
            if let Ok(entry) = res {
                let model = Model::from_file(entry.path())?;
                models.push(model);
            }
        }

        Ok(Self {
            models,
            db_uri,
            ..project
        })
    }

    pub fn set_db_uri(&mut self, uri: String) {
        self.db_uri = uri;
    }

    pub fn generate_template(&self) -> Result<()> {
        // Create project directories
        fs::create_dir(&self.name)?;
        let folders = vec!["public", "models", "public/scripts", "public/styles"];
        for folder in folders {
            fs::create_dir(format!("{}/{}", self.name, folder))?;
        }

        // Create model files
        for model in &self.models {
            let path = format!("{}/models/{}.json", self.name, model.name);
            fs::write(&path, serde_json::to_string(model).unwrap())?;
        }

        // Copy static files
        fs::copy(
            "template/scripts/feather.min.js",
            format!("{}/public/scripts/feather.min.js", self.name),
        )?;

        // Copy compiled files
        for path in Compiled::iter() {
            let file = Compiled::get(&path).unwrap();
            let content = std::str::from_utf8(&file.data)?;
            fs::write(format!("{}/public/{}", self.name, &path), content)?;
        }

        // Misc
        fs::write(
            format!("{}/.env", self.name),
            format!("DATABASE_URI={}", self.db_uri),
        )?;

        fs::write(format!("{}/.gitignore", self.name), ".env")?;

        fs::write(
            format!("{}/mezzo.json", self.name),
            serde_json::to_string(self).unwrap(),
        )?;

        Ok(())
    }
}
