use anyhow::Result;
use handlebars::{Context, Handlebars, Helper, HelperResult, Output, RenderContext};
use serde::{Deserialize, Serialize};
use serde_json::json;

use crate::{
    embedded::{Pages, Parts},
    model::{Field, Model},
    project::Project,
};

#[derive(Serialize, Deserialize)]
struct TemplateContext<T> {
    project: Project,
    models: Vec<Model>,
    context: T,
}

fn json_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value();
    out.write(&serde_json::to_string_pretty(param).unwrap())?;
    Ok(())
}

fn title_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value().as_str().unwrap();
    let mut chars = param.chars();

    let result = match chars.next() {
        None => String::new(),
        Some(c) => c.to_uppercase().collect::<String>() + chars.as_str(),
    };

    out.write(&result)?;
    Ok(())
}

fn form_field_helper(
    h: &Helper,
    hb: &Handlebars,
    _ctx: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let field = serde_json::from_value::<Field>(h.param(0).unwrap().value().to_owned())?;

    let field_template = format!("form/{}_field", json!(field.content_type).as_str().unwrap());

    match hb.render(&field_template, &json!(field)) {
        Ok(field) => out.write(&field)?,
        Err(err) => out.write(&format!("<span class=\"error\">{}</span>", err.desc))?,
    }

    Ok(())
}

pub fn init_handlebars() -> Result<Handlebars<'static>> {
    let mut hb = Handlebars::new();

    // Register helpers
    hb.register_helper("json", Box::new(json_helper));
    hb.register_helper("title", Box::new(title_helper));
    hb.register_helper("form_field", Box::new(form_field_helper));

    // Register parts
    for path in Parts::iter() {
        let name = path.to_string().replace(".hbs", "");
        if let Some(file) = Parts::get(&path) {
            let raw = String::from_utf8(file.data.to_vec())?;
            hb.register_partial(&name, raw)?;
        }
    }

    // Register pages
    for path in Pages::iter() {
        let name = path.to_string().replace(".hbs", "");
        if let Some(file) = Pages::get(&path) {
            let raw = String::from_utf8(file.data.to_vec())?;
            hb.register_template_string(&name, &raw)?;
        }
    }

    Ok(hb)
}
