use std::{fs, sync::Arc};

use actix_web::{get, post, web, Responder};
use serde_json::json;

use crate::{
    api::ApiResult,
    model::Model,
    server::{
        utils::{model::get_model, render::render_page},
        AppContext,
    },
};

#[get("/models/new")]
pub async fn create_model_view(context: web::Data<AppContext>) -> impl Responder {
    render_page("create_model", context.into_inner(), json!({}))
}

#[post("/models/new")]
pub async fn create_model(
    body: web::Json<Model>,
    context: web::Data<AppContext>,
) -> impl Responder {
    let model = body.into_inner();
    if model.name.trim().len() > 0 {
        let project = &context.project;
        let path = format!("{}/models/{}.json", project.name, model.name);
        fs::write(&path, serde_json::to_string_pretty(&model).unwrap()).unwrap();
        web::Json(json!({ "status": "ok" }))
    } else {
        web::Json(json!({ "status": "error" }))
    }
}

#[get("/models/{model_name}")]
pub async fn model_details(
    web::Path(model_name): web::Path<String>,
    context: web::Data<AppContext>,
) -> impl Responder {
    if let Some(model) = get_model(Arc::clone(&context), model_name) {
        let data = match context.client.find_all(&model.name).await.unwrap() {
            ApiResult::Data(data) => Some(data),
            _ => None,
        };

        render_page(
            "model",
            Arc::clone(&context),
            json!({ "model": model, "entities": data }),
        )
    } else {
        render_page("404", Arc::clone(&context), json!({ "thing": "Model" }))
    }
}
