use crate::{api::ApiResult, server::AppContext};
use actix_web::{delete, get, post, put, web, Responder};
use mongodb::bson::Document;
use serde_json::json;

#[get("/api")]
async fn root(context: web::Data<AppContext>) -> impl Responder {
    web::Json(
        json!({ "api": context.project.name, "version": context.project.version, "models": context.project.models }),
    )
}

#[get("/api/{content_type}")]
async fn all(
    web::Path(content_type): web::Path<String>,
    context: web::Data<AppContext>,
) -> impl Responder {
    match context.client.find_all(&content_type).await {
        Ok(data) => web::Json(data),
        Err(err) => web::Json(ApiResult::Err(format!("{:?}", err))),
    }
}

#[get("/api/{content_type}/{id}")]
async fn by_id(
    web::Path((content_type, id)): web::Path<(String, String)>,
    context: web::Data<AppContext>,
) -> impl Responder {
    match context.client.find_by_id(&content_type, &id).await {
        Ok(data) => web::Json(data),
        Err(err) => web::Json(ApiResult::Err(format!("{:?}", err))),
    }
}

#[post("/api/{content_type}")]
async fn create(
    web::Path(content_type): web::Path<String>,
    context: web::Data<AppContext>,
    data: web::Json<Document>,
) -> impl Responder {
    match context
        .client
        .create(&content_type, data.into_inner())
        .await
    {
        Ok(data) => web::Json(data),
        Err(err) => web::Json(ApiResult::Err(format!("{:?}", err))),
    }
}

#[put("/api/{content_type}/{id}")]
async fn update(
    web::Path((content_type, id)): web::Path<(String, String)>,
    context: web::Data<AppContext>,
    data: web::Json<Document>,
) -> impl Responder {
    match context
        .client
        .update(&content_type, &id, data.into_inner())
        .await
    {
        Ok(data) => web::Json(data),
        Err(err) => web::Json(ApiResult::Err(format!("{:?}", err))),
    }
}

#[delete("/api/{content_type}/{id}")]
async fn remove(
    web::Path((content_type, id)): web::Path<(String, String)>,
    context: web::Data<AppContext>,
) -> impl Responder {
    match context.client.remove(&content_type, &id).await {
        Ok(data) => web::Json(data),
        Err(err) => web::Json(ApiResult::Err(format!("{:?}", err))),
    }
}
