use anyhow::Result;
use handlebars::{Context, Handlebars, Helper, HelperResult, Output, RenderContext};
use rust_embed::RustEmbed;

#[derive(RustEmbed)]
#[folder = "template/parts/"]
pub struct Parts;

#[derive(RustEmbed)]
#[folder = "template/pages/"]
pub struct Pages;

#[derive(RustEmbed)]
#[folder = "out"]
pub struct Compiled;

fn json_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value();
    out.write(&serde_json::to_string_pretty(param).unwrap())?;
    Ok(())
}

fn title_helper(
    h: &Helper,
    _: &Handlebars,
    _: &Context,
    _rc: &mut RenderContext,
    out: &mut dyn Output,
) -> HelperResult {
    let param = h.param(0).unwrap().value().as_str().unwrap();
    let mut chars = param.chars();

    let result = match chars.next() {
        None => String::new(),
        Some(c) => c.to_uppercase().collect::<String>() + chars.as_str(),
    };

    out.write(&result)?;
    Ok(())
}

pub fn init_handlebars() -> Result<Handlebars<'static>> {
    let mut hb = Handlebars::new();

    // Register helpers
    hb.register_helper("json", Box::new(json_helper));
    hb.register_helper("title", Box::new(title_helper));

    // Register parts
    for path in Parts::iter() {
        let name = path.to_string().replace(".hbs", "");
        if let Some(file) = Parts::get(&path) {
            let raw = String::from_utf8(file.data.to_vec())?;
            hb.register_partial(&name, raw)?;
        }
    }

    // Register pages
    for path in Pages::iter() {
        let name = path.to_string().replace(".hbs", "");
        if let Some(file) = Pages::get(&path) {
            let raw = String::from_utf8(file.data.to_vec())?;
            hb.register_template_string(&name, &raw)?;
        }
    }

    Ok(hb)
}
