use anyhow::Result;
use regex::Regex;

use crate::template::Styles;

const IMPORT_REGEX: &str = r"@import '(.*)';";

fn extract_imports(file: &String) -> (String, Vec<String>) {
    let lines = file.lines();
    let regex = Regex::new(IMPORT_REGEX).unwrap();
    let stmts = lines.filter(|line| regex.is_match(line));

    let new_file = regex.replace_all(file, "");

    let imports = stmts
        .map(|import| {
            let path = regex.captures(import).unwrap().get(1).unwrap().as_str();
            format!("{}.scss", path)
        })
        .collect();

    (String::from(new_file), imports)
}

pub fn compile_styles() -> Result<String> {
    let main = Styles::get("main.scss").unwrap();
    let raw_scss = String::from_utf8(main.data.to_vec())?;

    let (mut scss, imports) = extract_imports(&raw_scss);

    for import in &imports {
        let file = Styles::get(import).unwrap();
        let part = std::str::from_utf8(&file.data)?;

        scss.push_str(&format!("\n{}", part));
    }

    let css = grass::from_string(scss, &grass::Options::default()).unwrap();

    Ok(css)
}
