use anyhow::Result;

use crate::{
    args::arg_or_prompt,
    model::{Field, Model},
    project::Project,
    utils::io::prompt,
};

pub fn create_project(args: Vec<String>) -> Result<()> {
    let project_name = arg_or_prompt(args.get(0), "Give your project a name: ");

    let mut project = Project::default(&project_name);

    let mut author_model = Model::new("author");
    author_model.add_field(Field::text("name"));

    let mut post_model = Model::new("post");
    post_model.add_field(Field::text("title"));
    post_model.add_field(Field::text("date"));
    post_model.add_field(Field::image("cover"));
    post_model.add_field(Field::reference("author", &author_model));

    let db_uri = prompt(
        "I am kindly asking for you mongo db uri: ",
        "mongodb://localhost:27017/mezzo",
    );

    project.set_db_uri(db_uri);

    project.models.push(post_model);
    project.models.push(author_model);

    project.generate_template()?;

    println!("\n✨ {} is ready!", project.name);
    Ok(())
}

#[cfg(test)]
mod create_project_tests {
    use std::{fs::remove_dir_all, path::Path};

    use super::*;

    #[test]
    fn test_create_project() {
        let project = Project::default("Alma");
        let path = Path::new(&project.name);
        assert!(!path.exists());
        project.generate_template().unwrap();
        assert!(&path.exists());

        dotenv::from_path(format!("{}/.env", project.name)).ok();

        let project_from_file = Project::from_dir(path.to_str().unwrap()).unwrap();
        assert_eq!(project_from_file.name, project.name);
        assert_eq!(project_from_file.version, project.version);
        remove_dir_all(path).unwrap();
        assert!(!path.exists());
    }
}
