mod routes;
mod utils;

use actix_web::{middleware::Logger, App, HttpServer};
use anyhow::{Context, Result};
use env_logger;
use handlebars::Handlebars;
use routes::api;
use routes::dashboard;

use crate::api::Client;
use crate::db::connect;
use crate::project::Project;
use crate::server::routes::models;
use crate::template;

pub struct AppContext {
    pub root: String,
    pub client: Client,
    pub project: Project,
    pub hb: Handlebars<'static>,
}

#[actix_rt::main]
pub async fn start_server(root: String) -> Result<()> {
    dotenv::from_path(format!("{}/.env", &root)).ok();
    println!("Starting server...");

    let host = "127.0.0.1";
    let port = 8000;

    env_logger::init_from_env(env_logger::Env::new().default_filter_or("info"));

    let hb = template::init_handlebars()?;

    println!("Loading configuration...");
    let project = Project::from_dir(&root)
        .context("Config file was not found. Are you sure this is a mezzo project?")?;

    println!("Connecting to the database...");
    let db = connect(project.db_uri.as_str()).await?.database("mezzo");
    let client = Client::new(db);

    let server = HttpServer::new(move || {
        let context = AppContext {
            client: client.clone(),
            project: project.clone(),
            root: root.clone(),
            hb: hb.clone(),
        };

        return App::new()
            .wrap(Logger::new("%r %s"))
            .data(context)
            .service(models::create_model)
            .service(models::model_details)
            .service(api::root)
            .service(api::all)
            .service(api::by_id)
            .service(api::create)
            .service(api::update)
            .service(api::remove)
            .service(dashboard::index)
            .service(dashboard::home)
            .service(dashboard::static_file);
    })
    .bind((host, port))?;

    println!("\n🚀 Live on http://{}:{}", host, port);
    server.run().await?;

    Ok(())
}
