use metrics::{describe_gauge, gauge, Unit};
use procfs::process::Process;
use procfs::{page_size, ticks_per_second, ProcResult};

pub fn describe() {
    describe_gauge!(
        "process_cpu_seconds_total",
        Unit::Seconds,
        "Total user and system CPU time spent in seconds."
    );
    // TODO(feature): Getting the number of open file descriptors involves extra work
    //     and I don't care *that* much about the metric for now anyway.
    // describe_gauge!("process_open_fds", Unit::Count, "Number of open file descriptors.");
    // TODO(feature): Getting the max number of file descriptors involves reading the limits file,
    //     which is easy enough to implement but I don't care about the limit if there's no count of
    //     open FDs (and may as well reduce the amount of work the collector needs to do).
    // describe_gauge!("process_max_fds", Unit::Count, "Maximum number of open file descriptors.");
    describe_gauge!(
        "process_virtual_memory_bytes",
        Unit::Bytes,
        "Virtual memory size in bytes."
    );
    // TODO(feature): Couldn't immediately see where to get this one, and I don't usually rely on it.
    // describe_gauge!("process_virtual_memory_max_bytes", Unit::Bytes, "Maximum amount of virtual memory available in bytes.");
    describe_gauge!(
        "process_resident_memory_bytes",
        Unit::Bytes,
        "Resident memory size in bytes."
    );
    // TODO(feature): counting heap bytes presumably needs access to the allocator.
    //     I can see that breaking depending on which allocator is in use, so won't bother for now.
    // describe_gauge!("process_heap_bytes", Unit::Bytes, "Process heap size in bytes.");
    describe_gauge!(
        "process_start_time_seconds",
        Unit::Seconds,
        "Start time of the process since unix epoch in seconds."
    );
    describe_gauge!(
        "process_threads",
        Unit::Count,
        "Number of OS threads in the process."
    );
}

pub fn emit_now() -> ProcResult<()> {
    let boot_time_secs = procfs::boot_time_secs()?;

    let this_process = Process::myself()?;

    let total_time_ticks = this_process.stat.utime + this_process.stat.stime;
    let ticks_per_second = ticks_per_second()?;

    let rss_bytes = this_process.stat.rss * page_size()?;

    let start_time_secs =
        (this_process.stat.starttime as f64) / (ticks_per_second as f64) + boot_time_secs as f64;
    let total_time_secs = (total_time_ticks as f64) / (ticks_per_second as f64);

    let num_threads = this_process.stat.num_threads;

    gauge!("process_cpu_seconds_total", total_time_secs);
    // gauge!("process_open_fds", );
    // gauge!("process_max_fds", );
    gauge!(
        "process_virtual_memory_bytes",
        this_process.stat.vsize as f64
    );
    // gauge!("process_virtual_memory_max_bytes", );
    gauge!("process_resident_memory_bytes", rss_bytes as f64);
    // gauge!("process_heap_bytes", );
    gauge!("process_start_time_seconds", start_time_secs);
    gauge!("process_threads", num_threads as f64);

    Ok(())
}
