//! A collection of nature-inspired meta-heuristic algorithms.
//!
//! # Algorithm
//!
//! There are two traits [`Algorithm`](crate::utility::Algorithm) and
//! [`Setting`](crate::utility::Setting) ([`setting!`]).
//! The previous is used to design the optimization method,
//! and the latter is the setting interface.
//!
//! [`Solver`] is a simple interface for obtaining the solution, or analyzing the result.
//! This type allows you to use the API without importing any traits.
//!
//! All provided methods are listed in the module [`methods`].
//!
//! # Objective Function
//!
//! You can define your question as a objective function through implementing [`ObjFunc`],
//! and then the upper bound, lower bound, and objective function [`ObjFunc::fitness`] should be defined.
//!
//! The final answer is [`ObjFunc::result`], which is calculated from the design parameters.
//!
//! # Random Function
//!
//! This crate use 32bit PRNG algorithm to generate random value, before that,
//! a random seed is required.
//! The seed is generated by `getrandom`, please see its [support platform](getrandom#supported-targets).
//!
//! # Features
//!
//! + `std`: Default feature. Enable standard library function, such as timing and threading.
//! + `parallel`: Enable parallel function, let objective function running without ordered,
//!   uses [`std::thread::spawn`].
//!   Disable it for the platform that doesn't supported threading,
//!   or if your objective function is not complicate enough.
//!   This feature required `std`.
//! + `wasm`: Support for webassembly, especial for random seed generating.
//! + `libm`: If the standard library is not provided, some math functions might missing.
//!   This will disable some pre-implemented algorithms.
//!   However, there is a math library implemented in pure Rust, the name is same as `libm`.
//!   This feature can re-enable (or replace) the math functions by using the `libm` crate.
#![cfg_attr(doc_cfg, feature(doc_cfg))]
#![cfg_attr(not(feature = "std"), no_std)]
#![warn(missing_docs)]
extern crate alloc;
#[cfg(not(feature = "std"))]
extern crate core as std;

pub use crate::methods::*;
pub use crate::obj_func::ObjFunc;
pub use crate::report::Report;
pub use crate::solver::Solver;
pub use crate::utility::setting::{Adaptive, Task};

/// The setting expression, used to fill the algorithm options.
/// This is the syntax shorter than the regular constructor.
///
/// Please make sure all fields are visible.
///
/// This macro will use the construct pattern, something like:
///
/// ```
/// use metaheuristics_nature::{setting, utility::BasicSetting};
///
/// #[derive(Default, Debug, PartialEq)]
/// struct Ga {
///     base: BasicSetting,
///     mutate: f64,
/// }
///
/// let constructed = Ga {
///     base: BasicSetting {
///         pop_num: 200,
///         ..Default::default()
///     },
///     mutate: 0.9,
/// };
/// let from_macro = setting!(Ga {
///     +base: { pop_num: 200 },
///     mutate: 0.9,
/// });
/// assert_eq!(constructed, from_macro);
/// ```
///
/// In the above example, the "base" setting needs to add a plus sign "+" to indicate the base setting.
/// Otherwise, all fields are listed as usual.
///
/// The single tuple type ("inherited" option) can use following syntax:
///
/// ```
/// use metaheuristics_nature::{setting, utility::BasicSetting};
///
/// #[derive(Default, Debug, PartialEq)]
/// struct Ga(BasicSetting);
///
/// let constructed = Ga(BasicSetting {
///     pop_num: 200,
///     ..Default::default()
/// });
/// let from_macro = setting!(Ga(pop_num: 200));
/// assert_eq!(constructed, from_macro);
/// ```
#[macro_export]
macro_rules! setting {
    (@base $(, $field:ident: $value:expr)*) => {
        $crate::utility::BasicSetting {
            $($field: $value,)*
            ..Default::default()
        }
    };
    ($name:ident{$(+$base:ident: {$($base_field:ident: $base_value:expr),* $(,)?})? $(, $field:ident: $value:expr)* $(,)?}) => {
        $name {
            $($base: $crate::setting!(@base, $($base_field: $base_value),*),)?
            $($field: $value,)*
            ..Default::default()
        }
    };
    ($name:ident($($field:ident: $value:expr),* $(,)?)) => {
        $name($crate::setting!(@base, $($field: $value),*))
    };
}

pub mod methods;
mod obj_func;
pub mod random;
mod report;
mod solver;
#[cfg(test)]
mod tests;
pub mod thread_pool;
pub mod utility;
