use meshtext::{Glyph, MeshText};

mod util;

/// Use this test to plot a triangulation diagram for the specified letter.
//#[test]
#[allow(unused)]
fn plot_glyph_vertices() {
    let character = 'A';
    let font_data = include_bytes!("../assets/font/FiraMono-Regular.ttf");
    let mut generator = meshtext::MeshGenerator::new(font_data);
    let mesh: MeshText = generator
        .generate_glyph(character, true, None)
        .expect("Failed to generate text mesh for character.");

    let filename = format!("tests/glyph_plot_of_{}.png", character);
    util::plot_vertices(filename.as_str(), true, &mesh.vertices).unwrap();
}

/// Test if the triangulation and outline generation of a single glyph (in this
/// case the captial letter `A`) works as intended.
///
/// In the font used here, the letter `A` has no curves
/// and is only composed of straight lines.
#[test]
fn test_flat_glyph_lines() {
    let font_data = include_bytes!("../assets/font/FiraMono-Regular.ttf");
    let mut generator = meshtext::MeshGenerator::new(font_data);
    let mesh: MeshText = generator
        .generate_glyph('A', true, None)
        .expect("Failed to generate text mesh for character A.");

    #[rustfmt::skip]
    let expected: Vec<f32> = vec![
        0.342500001, 0.206666663, 0f32,
        0.153333336, 0.206666663, 0f32,
        0.134166673, 0.146666661, 0f32,
        0.342500001, 0.206666663, 0f32,
        0.134166673, 0.146666661, 0f32,
        0.361666679, 0.146666661, 0f32,
        0.153333336, 0.206666663, 0f32,
        0.204166666, 0.574166656, 0f32,
        0.0166666675, 0f32, 0f32,
        0.134166673, 0.146666661, 0f32,
        0.153333336, 0.206666663, 0f32,
        0.0166666675, 0f32, 0f32,
        0.153333336, 0.206666663, 0f32,
        0.248333335, 0.512499988, 0f32,
        0.204166666, 0.574166656, 0f32,
        0.342500001, 0.206666663, 0f32,
        0.295833319, 0.574166656, 0f32,
        0.248333335, 0.512499988, 0f32,
        0.342500001, 0.206666663, 0f32,
        0.361666679, 0.146666661, 0f32,
        0.482499987, 0f32, 0f32,
        0.295833319, 0.574166656, 0f32,
        0.342500001, 0.206666663, 0f32,
        0.482499987, 0f32, 0f32,
        0.295833319, 0.574166656, 0f32,
        0.204166666, 0.574166656, 0f32,
        0.248333335, 0.512499988, 0f32,
        0.361666679, 0.146666661, 0f32,
        0.407499999, 0f32, 0f32,
        0.482499987, 0f32, 0f32,
        0.134166673, 0.146666661, 0f32,
        0.0166666675, 0f32, 0f32,
        0.088333331, 0f32, 0f32,
    ];

    assert_eq!(expected, mesh.vertices);
}

/// Test if the triangulation and outline generation of a single glyph (in this
/// case the captial letter `A`) works as intended in the two-dimensional case.
///
/// In the font used here, the letter `A` has no curves
/// and is only composed of straight lines.
#[test]
fn test_flat_glyph_lines_2d() {
    let font_data = include_bytes!("../assets/font/FiraMono-Regular.ttf");
    let mut generator = meshtext::MeshGenerator::new(font_data);
    let mesh: MeshText = generator
        .generate_glyph_2d('A', Some(&glam::Mat3::IDENTITY.to_cols_array()))
        .expect("Failed to generate text mesh for character A.");

    #[rustfmt::skip]
    let expected: Vec<f32> = vec![
        0.342500001, 0.206666663,
        0.153333336, 0.206666663,
        0.134166673, 0.146666661,
        0.342500001, 0.206666663,
        0.134166673, 0.146666661,
        0.361666679, 0.146666661,
        0.153333336, 0.206666663,
        0.204166666, 0.574166656,
        0.0166666675, 0f32,
        0.134166673, 0.146666661,
        0.153333336, 0.206666663,
        0.0166666675, 0f32,
        0.153333336, 0.206666663,
        0.248333335, 0.512499988,
        0.204166666, 0.574166656,
        0.342500001, 0.206666663,
        0.295833319, 0.574166656,
        0.248333335, 0.512499988,
        0.342500001, 0.206666663,
        0.361666679, 0.146666661,
        0.482499987, 0f32,
        0.295833319, 0.574166656,
        0.342500001, 0.206666663,
        0.482499987, 0f32,
        0.295833319, 0.574166656,
        0.204166666, 0.574166656,
        0.248333335, 0.512499988,
        0.361666679, 0.146666661,
        0.407499999, 0f32,
        0.482499987, 0f32,
        0.134166673, 0.146666661,
        0.0166666675, 0f32,
        0.088333331, 0f32,
    ];

    assert_eq!(expected, mesh.vertices);
}

/// Test if the triangulation and outline generation of a single glyph (in this
/// case the captial letter `Ö`) works as intended.
///
/// In the font used here, the letter `Ö` is composed of quadratic bezier curves only.
#[test]
fn test_flat_glyph_quads() {
    let font_data = include_bytes!("../assets/font/FiraMono-Regular.ttf");
    let quality = meshtext::QualitySettings {
        quad_interpolation_steps: 3,
        cubic_interpolation_steps: 3,
    };
    let mut generator = meshtext::MeshGenerator::new_with_quality(font_data, quality);
    let mesh: MeshText = generator
        .generate_glyph('Ö', true, None)
        .expect("Failed to generate text mesh for character Ö.");

    #[rustfmt::skip]
    let expected: Vec<f32> = vec![
        0.1157716, 0.35260287, 0f32,
        0.1286111, 0.41675925, 0f32,
        0.050205763, 0.3998457, 0f32,
        0.1286111, 0.41675925, 0f32,
        0.06333333, 0.44083333, 0f32,
        0.050205763, 0.3998457, 0f32,
        0.1157716, 0.35260287, 0f32,
        0.050205763, 0.3998457, 0f32,
        0.041851852, 0.34194446, 0f32,
        0.11416667, 0.28583333, 0f32,
        0.1157716, 0.35260287, 0f32,
        0.041851852, 0.34194446, 0f32,
        0.11782407, 0.21449074, 0f32,
        0.11416667, 0.28583333, 0f32,
        0.039166667, 0.28583333, 0f32,
        0.1286111, 0.41675925, 0f32,
        0.08203704, 0.4825, 0f32,
        0.06333333, 0.44083333, 0f32,
        0.1286111, 0.41675925, 0f32,
        0.14666666, 0.46291667, 0f32,
        0.08203704, 0.4825, 0f32,
        0.14666666, 0.46291667, 0f32,
        0.16781893, 0.4952829, 0f32,
        0.108004116, 0.5221296, 0f32,
        0.14666666, 0.46291667, 0f32,
        0.108004116, 0.5221296, 0f32,
        0.08203704, 0.4825, 0f32,
        0.16781893, 0.4952829, 0f32,
        0.20703703, 0.51587963, 0f32,
        0.1688889, 0.56666666, 0f32,
        0.11416667, 0.28583333, 0f32,
        0.041851852, 0.34194446, 0f32,
        0.039166667, 0.28583333, 0f32,
        0.37098765, 0.42539093, 0f32,
        0.3822222, 0.3585185, 0f32,
        0.4365741, 0.44314814, 0f32,
        0.4365741, 0.44314814, 0f32,
        0.35333332, 0.46333334, 0f32,
        0.37098765, 0.42539093, 0f32,
        0.35333332, 0.46333334, 0f32,
        0.40625, 0.5029167, 0f32,
        0.3274074, 0.49620372, 0f32,
        0.35333332, 0.46333334, 0f32,
        0.4365741, 0.44314814, 0f32,
        0.40625, 0.5029167, 0f32,
        0.3274074, 0.49620372, 0f32,
        0.40625, 0.5029167, 0f32,
        0.3715792, 0.5465998, 0f32,
        0.29008234, 0.5195782, 0f32,
        0.3274074, 0.49620372, 0f32,
        0.31254628, 0.57439816, 0f32,
        0.3822222, 0.3585185, 0f32,
        0.45813787, 0.36442384, 0f32,
        0.4365741, 0.44314814, 0f32,
        0.3822222, 0.3585185, 0f32,
        0.38583332, 0.28666666, 0f32,
        0.45813787, 0.36442384, 0f32,
        0.38583332, 0.28666666, 0f32,
        0.38422838, 0.21993828, 0f32,
        0.46083334, 0.28666666, 0f32,
        0.45813787, 0.36442384, 0f32,
        0.38583332, 0.28666666, 0f32,
        0.46083334, 0.28666666, 0f32,
        0.3274074, 0.49620372, 0f32,
        0.3715792, 0.5465998, 0f32,
        0.31254628, 0.57439816, 0f32,
        0.108004116, 0.5221296, 0f32,
        0.16781893, 0.4952829, 0f32,
        0.135, 0.54583335, 0f32,
        0.16781893, 0.4952829, 0f32,
        0.1688889, 0.56666666, 0f32,
        0.135, 0.54583335, 0f32,
        0.20703703, 0.51587963, 0f32,
        0.25, 0.5225, 0f32,
        0.21098766, 0.58148146, 0f32,
        0.21098766, 0.58148146, 0f32,
        0.25, 0.5225, 0f32,
        0.25, 0.5833333, 0f32,
        0.1688889, 0.56666666, 0f32,
        0.20703703, 0.51587963, 0f32,
        0.21098766, 0.58148146, 0f32,
        0.25, 0.5225, 0f32,
        0.29008234, 0.5195782, 0f32,
        0.25, 0.5833333, 0f32,
        0.25, 0.5833333, 0f32,
        0.29008234, 0.5195782, 0f32,
        0.31254628, 0.57439816, 0f32,
        0.38422838, 0.21993828, 0f32,
        0.45476854, 0.20314814, 0f32,
        0.46083334, 0.28666666, 0f32,
        0.38422838, 0.21993828, 0f32,
        0.37138888, 0.1561111, 0f32,
        0.45476854, 0.20314814, 0f32,
        0.11782407, 0.21449074, 0f32,
        0.041841563, 0.20768519, 0f32,
        0.06324074, 0.12916666, 0f32,
        0.12920268, 0.14809155, 0f32,
        0.11782407, 0.21449074, 0f32,
        0.06324074, 0.12916666, 0f32,
        0.11782407, 0.21449074, 0f32,
        0.039166667, 0.28583333, 0f32,
        0.041841563, 0.20768519, 0f32,
        0.12920268, 0.14809155, 0f32,
        0.06324074, 0.12916666, 0f32,
        0.14708333, 0.110416666, 0f32,
        0.37138888, 0.1561111, 0f32,
        0.4359002, 0.12220164, 0f32,
        0.45476854, 0.20314814, 0f32,
        0.37138888, 0.1561111, 0f32,
        0.35333332, 0.110416666, 0f32,
        0.4359002, 0.12220164, 0f32,
        0.14708333, 0.110416666, 0f32,
        0.06324074, 0.12916666, 0f32,
        0.093333334, 0.07, 0f32,
        0.17314816, 0.07777778, 0f32,
        0.14708333, 0.110416666, 0f32,
        0.093333334, 0.07, 0f32,
        0.18694444, 0.64555556, 0f32,
        0.16416667, 0.64, 0f32,
        0.17521605, 0.64061725, 0f32,
        0.32375, 0.6413889, 0f32,
        0.30375, 0.6525, 0f32,
        0.31171295, 0.6457099, 0f32,
        0.32375, 0.6413889, 0f32,
        0.33583334, 0.64, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.35333332, 0.110416666, 0f32,
        0.33218107, 0.07850309, 0f32,
        0.40625, 0.071666665, 0f32,
        0.30375, 0.6525, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.2965741, 0.6615741, 0f32,
        0.30375, 0.6525, 0f32,
        0.32375, 0.6413889, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.33583334, 0.64, 0f32,
        0.3469033, 0.64061725, 0f32,
        0.3587963, 0.64555556, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.19625, 0.6525, 0f32,
        0.20773147, 0.67212963, 0f32,
        0.16416667, 0.64, 0f32,
        0.18694444, 0.64555556, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.18694444, 0.64555556, 0f32,
        0.19625, 0.6525, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.20773147, 0.67212963, 0f32,
        0.19625, 0.6525, 0f32,
        0.20326646, 0.6602366, 0f32,
        0.15162037, 0.6413889, 0f32,
        0.16416667, 0.64, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.4359002, 0.12220164, 0f32,
        0.35333332, 0.110416666, 0f32,
        0.40625, 0.071666665, 0f32,
        0.33218107, 0.07850309, 0f32,
        0.29296297, 0.058194447, 0f32,
        0.36462963, 0.026759256, 0f32,
        0.33583334, 0.64, 0f32,
        0.3587963, 0.64555556, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.36833334, 0.6525, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.3587963, 0.64555556, 0f32,
        0.12407407, 0.6615741, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.11897119, 0.67313784, 0f32,
        0.13925411, 0.6457099, 0f32,
        0.15162037, 0.6413889, 0f32,
        0.13125, 0.6525, 0f32,
        0.17314816, 0.07777778, 0f32,
        0.093333334, 0.07, 0f32,
        0.12781893, 0.026995888, 0f32,
        0.2103498, 0.054567903, 0f32,
        0.17314816, 0.07777778, 0f32,
        0.18703704, -0.00037036976, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.12407407, 0.6615741, 0f32,
        0.13125, 0.6525, 0f32,
        0.2965741, 0.6615741, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.2914712, 0.67313784, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.36833334, 0.6525, 0f32,
        0.3801852, 0.67212963, 0f32,
        0.16416667, 0.64, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.13125, 0.6525, 0f32,
        0.15162037, 0.6413889, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.33583334, 0.64, 0f32,
        0.33583334, 0.7291667, 0f32,
        0.36833334, 0.6525, 0f32,
        0.37557614, 0.6602366, 0f32,
        0.3801852, 0.67212963, 0f32,
        0.40625, 0.071666665, 0f32,
        0.33218107, 0.07850309, 0f32,
        0.36462963, 0.026759256, 0f32,
        0.29296297, 0.058194447, 0f32,
        0.25, 0.051666666, 0f32,
        0.25, -0.009166666, 0f32,
        0.17314816, 0.07777778, 0f32,
        0.12781893, 0.026995888, 0f32,
        0.18703704, -0.00037036976, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.20852882, 0.69521606, 0f32,
        0.20342593, 0.7069444, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.30375, 0.71625, 0f32,
        0.2922685, 0.69625, 0f32,
        0.33583334, 0.64, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.36833334, 0.71625, 0f32,
        0.30375, 0.71625, 0f32,
        0.29083332, 0.68416667, 0f32,
        0.33583334, 0.7291667, 0f32,
        0.19625, 0.71625, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.20342593, 0.7069444, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.16416667, 0.64, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.30375, 0.71625, 0f32,
        0.33583334, 0.7291667, 0f32,
        0.31305555, 0.7234259, 0f32,
        0.20916666, 0.68416667, 0f32,
        0.19625, 0.71625, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.2922685, 0.69625, 0f32,
        0.30375, 0.71625, 0f32,
        0.29673356, 0.70828706, 0f32,
        0.36462963, 0.026759256, 0f32,
        0.29296297, 0.058194447, 0f32,
        0.3079218, -0.0051748976, 0f32,
        0.29296297, 0.058194447, 0f32,
        0.25, -0.009166666, 0f32,
        0.3079218, -0.0051748976, 0f32,
        0.25, 0.051666666, 0f32,
        0.2103498, 0.054567903, 0f32,
        0.25, -0.009166666, 0f32,
        0.2103498, 0.054567903, 0f32,
        0.18703704, -0.00037036976, 0f32,
        0.25, -0.009166666, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.38100824, 0.69521606, 0f32,
        0.37574074, 0.7069444, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.19625, 0.71625, 0f32,
        0.17625, 0.72773147, 0f32,
        0.36833334, 0.71625, 0f32,
        0.38166666, 0.68416667, 0f32,
        0.37574074, 0.7069444, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.13125, 0.71625, 0f32,
        0.11976852, 0.69625, 0f32,
        0.11833333, 0.68416667, 0f32,
        0.13125, 0.71625, 0f32,
        0.19625, 0.71625, 0f32,
        0.18828703, 0.7232664, 0f32,
        0.17625, 0.72773147, 0f32,
        0.33583334, 0.64, 0f32,
        0.36833334, 0.71625, 0f32,
        0.34796295, 0.72773147, 0f32,
        0.12423354, 0.70828706, 0f32,
        0.11976852, 0.69625, 0f32,
        0.13125, 0.71625, 0f32,
        0.33583334, 0.7291667, 0f32,
        0.33583334, 0.64, 0f32,
        0.34796295, 0.72773147, 0f32,
        0.31305555, 0.7234259, 0f32,
        0.33583334, 0.7291667, 0f32,
        0.32478395, 0.7285288, 0f32,
        0.13125, 0.71625, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.14064816, 0.7234259, 0f32,
        0.36833334, 0.71625, 0f32,
        0.36014402, 0.7232664, 0f32,
        0.34796295, 0.72773147, 0f32,
        0.14064816, 0.7234259, 0f32,
        0.16416667, 0.7291667, 0f32,
        0.1526646, 0.7285288, 0f32,
    ];

    assert_eq!(expected, mesh.vertices);
}
