/** A mergable map.
 * 
 * This map must contain a `Mergable` value. This determines the semantics on conflict. For example if the value is a `Cell` then last-write-wins semantics will be used. However if the value is a `Bag` then multiple values inserted into the same key will be kept. If the value is a `Counter` than conflicting values will be summed.
 */
#[derive(Clone)]
pub struct Map<K, V, Seq> {
	live: std::collections::HashMap<K, Value<V, Seq>>,
	removed: std::collections::HashMap<K, Seq>,
}

impl<
	K: Eq + std::hash::Hash,
	V: crate::Mergable,
	SF: crate::SequenceFactory,
>
	Map<K, V, crate::Sequence<SF>>
{
	pub fn entry<'a>(&'a mut self, key: K) -> MapEntry<'a, K, V, SF> {
		match self.live.entry(key) {
			std::collections::hash_map::Entry::Occupied(entry) => {
				MapEntry::Occupied(MapOccupiedEntry {
					entry,
					removed: &mut self.removed,
				})
			}
			std::collections::hash_map::Entry::Vacant(entry) => {
				MapEntry::Vacant(MapVacantEntry {
					entry,
				})
			}
		}
	}

	/** Insert a value into the map.
	 * 
	 * Note that if the value is already in the map the two values will be *merged*. If you do not want this to occur `delete` the value first.

	 * Also note that this updates the key's creation time meaning that deletes on the previous key will no longer be respected.

	 * TODO: Weird revival stuff.
	*/
	pub fn insert(&mut self, ctx: &mut crate::Context<SF>, k: K, v: V) {
		self.entry(k).insert(ctx, v);
	}

	/** Remove a value from the map.
	 *
	 * Note that the semantics of this operation can be confusing read the description carefully.
	 *
	 * This removes the *known* version of the element in the map and all older versions. Unknown modifications will not prevent deletion but if the element has been `insert`ed again after the version that we are aware of that insert will be preserved. Also note that due to merging semantics that may result in values from the known version "reappearing".
	 */
	pub fn remove<Q: ?Sized>(&mut self, k: K) -> Option<V>
		where
			K: std::borrow::Borrow<Q>,
			Q: std::hash::Hash + Eq
	{
		if let MapEntry::Occupied(entry) = self.entry(k) {
			Some(entry.remove())
		} else {
			None
		}
	}
}

impl<
	K: Clone + Eq + std::hash::Hash,
	V: crate::Mergable,
	SF: crate::SequenceFactory,
>
	crate::Mergable for Map<K, V, crate::Sequence<SF>>
{
	fn merge(&mut self, that: Self) {
		for (k, sequence) in that.removed {
			match self.removed.entry(k) {
				std::collections::hash_map::Entry::Occupied(mut removed_entry) => {
					if sequence > *removed_entry.get() {
						// std won't give my key back to me 😢
						let k = removed_entry.key().clone();

						let le = self.live.entry(k);
						if let std::collections::hash_map::Entry::Occupied(entry) = le {
							if entry.get().sequence <= sequence {
								entry.remove();
							}
						}

						removed_entry.insert(sequence);
					}
				}
				std::collections::hash_map::Entry::Vacant(entry) => {
					// std won't give my key back to me 😢
					let k = entry.key().clone();

					let le = self.live.entry(k);
					if let std::collections::hash_map::Entry::Occupied(entry) = le {
						if entry.get().sequence <= sequence {
							entry.remove();
						}
					}

					entry.insert(sequence);
				}
			}
		}

		for (k, Value{sequence, value}) in that.live {
			if let Some(seq) = self.removed.get(&k) {
				if seq >= &sequence {
					continue
				}
			}

			match self.entry(k) {
				MapEntry::Occupied(mut entry) => {
					let slot = entry.entry.get_mut();
					if sequence > slot.sequence {
						slot.sequence = sequence;
					}
					slot.value.merge(value);
				}
				MapEntry::Vacant(entry) => {
					entry.entry.insert(Value{sequence, value});
				}
			}
		}
	}
}

impl<
	K,
	V,
	SF,
>
  Default for Map<K, V, SF>
{
	fn default() -> Self {
		Map {
			live: Default::default(),
			removed: Default::default(),
		}
	}
}

impl<K: Eq + std::hash::Hash, V: PartialEq, Seq> PartialEq for Map<K, V, Seq> {
	fn eq(&self, that: &Self) -> bool {
		self.live == that.live
	}
}

impl<K: std::fmt::Debug, V: std::fmt::Debug, Seq> std::fmt::Debug for Map<K, V, Seq> {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		self.live.fmt(f)
	}
}

#[derive(Clone)]
struct Value<V, Seq> {
	sequence: Seq,
	value: V,
}

impl<V: PartialEq, Seq> PartialEq for Value<V, Seq> {
	fn eq(&self, that: &Self) -> bool {
		self.value == that.value
	}
}

impl<V: std::fmt::Debug, Seq> std::fmt::Debug for Value<V, Seq> {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		// write!(f, "{:?} @ {:?}", self.value, self.sequence)
		self.value.fmt(f)
	}
}

pub enum MapEntry<'a, K: 'a, V: 'a, SF: crate::SequenceFactory> {
	Occupied(MapOccupiedEntry<'a, K, V, SF>),
	Vacant(MapVacantEntry<'a, K, V, SF>),
}

impl<'a,
	K: 'a + Eq + std::hash::Hash,
	V: 'a + crate::Mergable,
	SF: crate::SequenceFactory
>
	MapEntry<'a, K, V, SF>
{
	pub fn insert(self, ctx: &'a mut crate::Context<SF>, v: V) -> &'a mut V {
		match self {
			MapEntry::Occupied(mut entry) => {
				entry.insert(ctx, v);
				entry.into_mut()
			}
			MapEntry::Vacant(entry) => entry.insert(ctx, v),
		}
	}

	pub fn or_insert(self, ctx: &'a mut crate::Context<SF>, v: V) -> &'a mut V {
		match self {
			MapEntry::Occupied(mut entry) => {
				entry.insert(ctx, v);
				entry.into_mut()
			}
			MapEntry::Vacant(entry) => entry.insert(ctx, v),
		}
	}
}

pub struct MapOccupiedEntry<'a, K: 'a, V: 'a, SF: crate::SequenceFactory> {
	entry: std::collections::hash_map::OccupiedEntry<'a, K, Value<V, crate::Sequence<SF>>>,
	removed: &'a mut std::collections::HashMap<K, crate::Sequence<SF>>,
}

impl<'a,
	K: 'a + Eq + std::hash::Hash,
	V: 'a + crate::Mergable,
	SF: crate::SequenceFactory
>
	MapOccupiedEntry<'a, K, V, SF>
{
	pub fn into_mut(self) -> &'a mut V {
		&mut self.entry.into_mut().value
	}

	pub fn insert(&mut self, ctx: &'a mut crate::Context<SF>, v: V) {
		self.entry.get_mut().sequence = ctx.next_sequence();
		self.entry.get_mut().value.merge(v);
	}

	pub fn remove(self) -> V {
		let (k, Value{sequence, value}) = self.entry.remove_entry();
		self.removed.insert(k, sequence);
		value
	}
}

pub struct MapVacantEntry<'a, K: 'a, V: 'a, SF: crate::SequenceFactory> {
	entry: std::collections::hash_map::VacantEntry<'a, K, Value<V, crate::Sequence<SF>>>,
}

impl<'a, K: 'a, V: 'a, SF: crate::SequenceFactory> MapVacantEntry<'a, K, V, SF> {
	pub fn insert(self, ctx: &'a mut crate::Context<SF>, value: V) -> &'a mut V {
		&mut self.entry.insert(
			Value {
				sequence: ctx.next_sequence(),
				value,
			})
			.value
	}
}

#[test]
fn test_map() {
	let mut ctx = crate::Context::default();

	let one = crate::Cell::new(&mut ctx, "one");
	let two = crate::Cell::new(&mut ctx, "two");

	let result = crate::test::test_merge(&mut [one.clone(), two.clone()]);
	assert_eq!(*result, "two");

	let mut parent_l = Map::default();
	parent_l.insert(&mut ctx, 0, two.clone());

	let mut parent_r = Map::default();
	parent_r.insert(&mut ctx, 0, one.clone());

	let merged = crate::test::test_merge(&mut [parent_l.clone(), parent_r.clone()]);
	assert_eq!(merged, parent_l);

	let mut child_l = merged.clone();
	child_l.insert(&mut ctx, 1, one.clone());

	let mut child_r = merged;
	child_r.insert(&mut ctx, 2, two.clone());

	let merged = crate::test::test_merge(&mut [parent_l, parent_r, child_l, child_r]);
	let mut expected = Map::default();
	expected.insert(&mut ctx, 0, two.clone());
	expected.insert(&mut ctx, 1, one.clone());
	expected.insert(&mut ctx, 2, two.clone());
	assert_eq!(merged, expected);
}

#[test]
fn test_map_remove_old() {
	let mut ctx = crate::Context::default();

	let one = crate::Cell::new(&mut ctx, "one");
	let two = crate::Cell::new(&mut ctx, "two");

	let mut parent_l = Map::default();
	parent_l.insert(&mut ctx, 0, two.clone());

	let mut parent_r = Map::default();
	parent_r.insert(&mut ctx, 0, one.clone());

	// We remove the left entry. However the right is created after this.
	parent_l.remove(0);

	let merged = crate::test::test_merge(&mut [parent_l.clone(), parent_r.clone()]);
	assert_eq!(merged, parent_r);
}

#[test]
fn test_map_remove_new() {
	let mut ctx = crate::Context::default();

	let one = crate::Cell::new(&mut ctx, "one");
	let two = crate::Cell::new(&mut ctx, "two");

	let mut parent_l = Map::default();
	parent_l.insert(&mut ctx, 0, two.clone());

	let mut parent_r = Map::default();
	parent_r.insert(&mut ctx, 0, one.clone());

	// We remove the right entry. Since this was created later the removal wins.
	parent_r.remove(0);

	let merged = crate::test::test_merge(&mut [parent_l.clone(), parent_r.clone()]);
	assert_eq!(merged, parent_r);
}

#[test]
fn test_map_remove_merge_revive() {
	let mut ctx = crate::Context::default();

	let one = crate::Cell::new(&mut ctx, "one");
	let two = crate::Cell::new(&mut ctx, "two");

	let mut root = Map::default();
	root.insert(&mut ctx, 0, two.clone());

	// Note that this "insert" actually merges the cells, resulting in "two" winning because it is newer (even though the insert happened earlier.
	let mut child_insert = root.clone();
	child_insert.insert(&mut ctx, 0, one.clone());

	// This removes the value. However since it doesn't know about child_insert's insertion that is preserved and the "two" value is "revived".
	let mut child_remove = root.clone();
	child_remove.remove(0);

	let merged = crate::test::test_merge(&mut [child_remove, child_insert.clone()]);
	assert_eq!(merged, child_insert);
}
