use serde::{Deserialize, Serialize};
use serde_aux::prelude::*;

use crate::common_types::{BackUrls, CheckoutProPayer, Item, PaymentMethods, Shipments};

#[derive(Debug, Clone, PartialEq, Serialize, Deserialize)]
pub struct CheckoutProPreferencesResponse {
    #[serde(deserialize_with = "deserialize_number_from_string")]
    pub client_id: i64,
    pub collector_id: i64,

    pub items: Vec<Item>,
    pub payer: CheckoutProPayer,
    pub back_urls: BackUrls,
    pub payment_methods: PaymentMethods,
    pub marketplace: String,
    pub marketplace_fee: i64,
    pub shipments: Shipments,

    /// Description that the payment will appear with in the card statement.
    pub statement_descriptor: Option<String>,

    #[serde(with = "time::serde::rfc3339")]
    pub date_created: time::OffsetDateTime,

    /// Autogenerated unique ID that identifies the preference.
    /// For example: 036151801-2484cd71-7140-4c51-985a-d4cfcf133baf
    #[serde(rename = "id")]
    pub preference_id: String,

    /// Automatically generated URL to open the Checkout.
    #[serde(rename = "init_point")]
    pub checkout_url: String,

    /// Automatically generated URL to open the Checkout in sandbox mode. Real users are used here,
    /// but transactions are executed using test credentials.
    #[serde(rename = "sandbox_init_point")]
    pub checkout_sandbox_url: String,

    /// Website locale. Example: MLB
    pub site_id: Option<String>,

    /// Valid JSON that can be added to the payment to save additional attributes
    pub metadata: Option<serde_json::Value>,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn t_deserialization() {
        let response = serde_json::from_slice::<CheckoutProPreferencesResponse>(include_bytes!(
            "../../tests/assets/checkout_preferences_response.json"
        ));

        assert!(response.is_ok())
    }
}
