// take care of these later ...
#![allow(clippy::new_without_default)]
#![allow(clippy::needless_lifetimes)]
#![allow(clippy::large_enum_variant)]
#![allow(clippy::type_complexity)]

pub mod builtin_types;
pub mod commands;
pub mod cyc_parser;
pub mod editor;
pub mod event;
pub mod event_helpers;
pub mod generator;
pub mod generator_processor;
pub mod interpreter;
pub mod markov_sequence_generator;
pub mod music_theory;
pub mod parameter;
pub mod parser;
pub mod pfa_growth;
pub mod pfa_reverse;
pub mod repl;
pub mod sample_set;
pub mod scheduler;
pub mod session;

use crate::builtin_types::*;
use crate::sample_set::SampleSet;
use crate::session::{OutputMode, Session};
use cpal::traits::{DeviceTrait, HostTrait, StreamTrait};
use directories_next::ProjectDirs;
use getopts::Options;
use parking_lot::Mutex;
use ruffbox_synth::ruffbox::{ReverbMode, Ruffbox};
use std::{env, sync, thread};

fn print_help(program: &str, opts: Options) {
    let description = format!(
        "{prog}: a markov-chain music language

Mégra is a DSL to make music with markov chains.

Usage:
    {prog} [options] [FILES...]
      ",
        prog = program,
    );
    println!("{}", opts.usage(&description));
}

// change this if you need to compile for really unreliable systems that need
// a lot of latency
// will be ignored if ringbuffer feature isn't activated
#[cfg(feature = "ringbuffer")]
const RINGBUFFER_SIZE: usize = 8000;
#[cfg(feature = "ringbuffer")]
const BLOCKSIZE: usize = 128;

fn main() -> Result<(), anyhow::Error> {
    let mut argv = env::args();
    let program = argv.next().unwrap();

    let mut opts = Options::new();
    opts.optflag("v", "version", "Print version");
    opts.optflag(
        "r",
        "repl",
        "no editor, repl only (i.e. for integration with other editors)",
    );
    opts.optflag("h", "help", "Print this help");
    opts.optflag("n", "no-samples", "don't load default samples");
    opts.optopt("o", "output-mode", "output mode (stereo, 8ch)", "stereo");
    opts.optflag("l", "list-devices", "list available devices");
    opts.optopt("d", "device", "choose device", "default");
    opts.optopt(
        "",
        "reverb-mode",
        "reverb mode (freeverb or convolution)",
        "freeverb",
    );
    opts.optopt(
        "",
        "font",
        "editor font (ComicMono, mononoki or custom path)",
        "mononoki",
    );
    opts.optopt("", "reverb-ir", "reverb impulse response (file)", "");
    opts.optopt(
        "",
        "live-buffer-time",
        "the capacity of the live input buffer in seconds",
        "3.0",
    );

    let matches = match opts.parse(argv) {
        Ok(m) => m,
        Err(e) => {
            eprintln!("Error: {}. Please see --help for more details", e);
            return Ok(());
        }
    };

    if matches.opt_present("v") {
        println!("0.0.3");
        return Ok(());
    }

    let editor: bool = !matches.opt_present("r");
    let load_samples: bool = !matches.opt_present("n");

    if matches.opt_present("h") {
        print_help(&program, opts);
        return Ok(());
    }

    let out_mode = match matches.opt_str("o").as_deref() {
        Some("8ch") => OutputMode::EightChannel,
        Some("4ch") => OutputMode::FourChannel,
        Some("stereo") => OutputMode::Stereo,
        _ => {
            println!("invalid output mode, assume stereo");
            OutputMode::Stereo
        }
    };

    let reverb_mode = match matches.opt_str("reverb-mode").as_deref() {
        Some("freeverb") => ReverbMode::FreeVerb,
        Some("convolution") => {
            match matches.opt_str("reverb-ir").as_deref() {
                Some(filepath) => {
                    if let Ok(mut reader) = claxon::FlacReader::open(filepath) {
                        let mut sample_buffer: Vec<f32> = Vec::new();
                        // decode to f32
                        let max_val =
                            (i32::MAX >> (32 - reader.streaminfo().bits_per_sample)) as f32;

                        for sample in reader.samples() {
                            let s = sample.unwrap() as f32 / max_val;
                            sample_buffer.push(s);
                        }
                        ReverbMode::Convolution(
                            sample_buffer,
                            reader.streaminfo().sample_rate as f32,
                        )
                    } else {
                        println!("reverb ir path invalid, fall back to freeverb");
                        ReverbMode::FreeVerb
                    }
                }
                None => {
                    println!("no reverb ir provided, fall back to freeverb");
                    ReverbMode::FreeVerb
                }
            }
        }
        _ => ReverbMode::FreeVerb,
    };

    let live_buffer_time: f32 = if let Some(s) = matches.opt_str("live-buffer-time") {
        if let Ok(f) = s.parse() {
            f
        } else {
            3.0
        }
    } else {
        3.0
    };

    println!("using a live buffer time of: {}", live_buffer_time);

    #[cfg(any(target_os = "linux", target_os = "dragonfly", target_os = "freebsd"))]
    let host = cpal::host_from_id(cpal::available_hosts()
				  .into_iter()
				  .find(|id| *id == cpal::HostId::Jack)
				  .expect(
				      "make sure --features jack is specified. only works on OSes where jack is available",
				  )).expect("jack host unavailable");

    #[cfg(not(any(target_os = "linux", target_os = "dragonfly", target_os = "freebsd")))]
    let host = cpal::default_host();

    if matches.opt_present("l") {
        for dev in host.output_devices()? {
            println!("{:?}", dev.name());
        }
        return Ok(());
    }

    let out_device = if let Some(dev) = matches.opt_str("d") {
        dev
    } else {
        "default".to_string()
    };

    let output_device = if out_device == "default" {
        host.default_output_device()
    } else {
        host.output_devices()?
            .find(|x| x.name().map(|y| y == out_device).unwrap_or(false))
    }
    .expect("failed to find output device");

    let input_device = if out_device == "default" {
        host.default_input_device()
    } else {
        host.input_devices()?
            .find(|x| x.name().map(|y| y == out_device).unwrap_or(false))
    }
    .expect("failed to find input device");

    println!("odev {}", output_device.name().unwrap());

    let out_config: cpal::SupportedStreamConfig = output_device.default_output_config().unwrap();
    let in_config: cpal::SupportedStreamConfig = input_device.default_input_config().unwrap();
    println!("in chan: {:?}", in_config);
    println!("out chan: {:?}", out_config);

    // let's assume it's the same for both ...
    let sample_format = out_config.sample_format();

    match out_mode {
        OutputMode::Stereo => {
            let mut out_conf: cpal::StreamConfig = out_config.into();
            let in_conf: cpal::StreamConfig = in_config.into();
            out_conf.channels = 2;
            match sample_format {
                cpal::SampleFormat::F32 => run::<f32, 2>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::I16 => run::<i16, 2>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::U16 => run::<u16, 2>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
            }
        }
        OutputMode::FourChannel => {
            let mut out_conf: cpal::StreamConfig = out_config.into();
            let in_conf: cpal::StreamConfig = in_config.into();
            out_conf.channels = 4;
            match sample_format {
                cpal::SampleFormat::F32 => run::<f32, 4>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::I16 => run::<i16, 4>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::U16 => run::<u16, 4>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
            }
        }
        OutputMode::EightChannel => {
            let mut out_conf: cpal::StreamConfig = out_config.into();
            let in_conf: cpal::StreamConfig = in_config.into();
            out_conf.channels = 8;
            match sample_format {
                cpal::SampleFormat::F32 => run::<f32, 8>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::I16 => run::<i16, 8>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
                cpal::SampleFormat::U16 => run::<u16, 8>(
                    &input_device,
                    &output_device,
                    &out_conf,
                    &in_conf,
                    out_mode,
                    live_buffer_time,
                    editor,
                    load_samples,
                    &reverb_mode,
                    matches.opt_str("font").as_deref(),
                )?,
            }
        }
    }

    Ok(())
}

#[allow(clippy::too_many_arguments)]
fn run<T, const NCHAN: usize>(
    input_device: &cpal::Device,
    output_device: &cpal::Device,
    out_config: &cpal::StreamConfig,
    in_config: &cpal::StreamConfig,
    mode: OutputMode,
    live_buffer_time: f32,
    editor: bool,
    load_samples: bool,
    reverb_mode: &ReverbMode,
    font: Option<&str>,
) -> Result<(), anyhow::Error>
where
    T: cpal::Sample,
{
    // at some point i'll need to implement more samplerates i suppose ...
    let sample_rate = out_config.sample_rate.0 as f32;
    let out_channels = out_config.channels as usize;
    let in_channels = in_config.channels as usize;
    let err_fn = |err| eprintln!("an error occurred on stream: {}", err);

    #[cfg(feature = "ringbuffer")]
    let ruffbox = sync::Arc::new(Mutex::new(Ruffbox::<128, NCHAN>::new(
        true,
        live_buffer_time.into(),
        reverb_mode,
        sample_rate.into(),
    )));

    #[cfg(not(feature = "ringbuffer"))]
    let ruffbox = sync::Arc::new(Mutex::new(Ruffbox::<512, NCHAN>::new(
        true,
        live_buffer_time.into(),
        reverb_mode,
        sample_rate.into(),
    )));

    let ruffbox2 = sync::Arc::clone(&ruffbox); // the one for the audio thread (out stream)...
    let ruffbox3 = sync::Arc::clone(&ruffbox); // the one for the audio thread (in stream)...

    let in_stream = input_device.build_input_stream(
        in_config,
        move |data: &[f32], _: &cpal::InputCallbackInfo| {
            let mut ruff = ruffbox3.lock();

            // there might be a faster way to de-interleave here ...
            // only use first input channel
            for (_, frame) in data.chunks(in_channels).enumerate() {
                ruff.write_sample_to_live_buffer(frame[0]);
            }
        },
        err_fn,
    )?;

    // main audio callback (plain)
    // the plain audio callback for platforms where the blocksize
    // is static, configurable and a power of two (preferably 512)
    // (i.e. jack, coreaudio)
    #[cfg(not(feature = "ringbuffer"))]
    let out_stream = output_device.build_output_stream(
        out_config,
        move |data: &mut [f32], _: &cpal::OutputCallbackInfo| {
            let mut ruff = ruffbox2.lock();

            // as the jack timing from cpal can't be trusted right now, the
            // ruffbox handles it's own logical time ...
            let ruff_out = ruff.process(0.0, true);

            // there might be a faster way to de-interleave here ...
            for (frame_count, frame) in data.chunks_mut(out_channels).enumerate() {
                for ch in 0..out_channels {
                    frame[ch] = ruff_out[ch][frame_count];
                }
            }
        },
        err_fn,
    )?;

    // main audio callback (with )
    // this is the ringbuffer version that internally buffers the audio
    // stream to allow a fixed blocksize being used for the ruffbox synth
    // even if the system doesn't provide a fixed and/or configurable blocksize
    // might require a higher latency
    #[cfg(feature = "ringbuffer")]
    println!("using ringbuffer to adapt blocksize, you might need to use a higher latency");

    #[cfg(feature = "ringbuffer")]
    let mut ringbuffer: [[f32; RINGBUFFER_SIZE]; NCHAN] = [[0.0; RINGBUFFER_SIZE]; NCHAN];
    #[cfg(feature = "ringbuffer")]
    let mut write_idx: usize = 0;
    #[cfg(feature = "ringbuffer")]
    let mut read_idx: usize = 0;
    #[cfg(feature = "ringbuffer")]
    let out_stream = output_device.build_output_stream(
        out_config,
        move |data: &mut [f32], _: &cpal::OutputCallbackInfo| {
            let mut ruff = ruffbox2.lock();

            let samples_available = if write_idx < read_idx {
                write_idx + RINGBUFFER_SIZE - read_idx
            } else if write_idx > read_idx {
                write_idx - read_idx
            } else {
                0
            };
            //let mut produced:usize = 0;

            let current_blocksize = data.len() / out_channels;

            //println!(
            //   "av {} need {} read {} write {}",
            //   samples_available, current_blocksize, read_idx, write_idx
            //);

            if samples_available < current_blocksize {
                let mut samples_actually_needed = current_blocksize - samples_available;

                while samples_actually_needed > 0 {
                    let ruff_out = ruff.process(0.0, true);
                    //produced += BLOCKSIZE;
                    for ch in 0..out_channels {
                        let mut tmp_write_idx = write_idx;
                        for s in 0..BLOCKSIZE {
                            ringbuffer[ch][tmp_write_idx] = ruff_out[ch][s];
                            tmp_write_idx += 1;
                            if tmp_write_idx >= RINGBUFFER_SIZE {
                                tmp_write_idx = 0;
                            }
                        }
                    }

                    write_idx += BLOCKSIZE;
                    if write_idx >= RINGBUFFER_SIZE {
                        write_idx = write_idx - RINGBUFFER_SIZE;
                    }

                    samples_actually_needed = if samples_actually_needed > BLOCKSIZE {
                        samples_actually_needed - BLOCKSIZE
                    } else {
                        0
                    }
                }
            }

            // there might be a faster way to de-interleave here ...
            for (_, frame) in data.chunks_mut(out_channels).enumerate() {
                for ch in 0..out_channels {
                    frame[ch] = ringbuffer[ch][read_idx];
                }
                read_idx += 1;
                if read_idx >= RINGBUFFER_SIZE {
                    read_idx = 0;
                }
            }
            /*
                println!(
                "POST BLOCK av {} bs {} to prod {} prod {} r idx {} w idx {} NOW {}",
                samples_available,
                current_blocksize,
                current_blocksize - samples_available,
                produced,
                read_idx,
                write_idx,
                ruff.get_now()
            );
                 */
        },
        err_fn,
    )?;

    in_stream.play()?;
    out_stream.play()?;

    // global data
    let session = sync::Arc::new(Mutex::new(Session::new()));
    let global_parameters = sync::Arc::new(GlobalParameters::with_capacity(1));
    let sample_set = sync::Arc::new(Mutex::new(SampleSet::new()));
    let parts_store = sync::Arc::new(Mutex::new(PartsStore::new()));

    // load the default sample set ...
    if load_samples {
        if let Some(proj_dirs) = ProjectDirs::from("de", "parkellipsen", "megra") {
            if !proj_dirs.config_dir().exists() {
                println!(
                    "create megra resource directory {:?}",
                    proj_dirs.config_dir()
                );
                std::fs::create_dir_all(proj_dirs.config_dir().to_str().unwrap())?;
            }

            let samples_path = proj_dirs.config_dir().join("samples");
            if !samples_path.exists() {
                println!("create megra samples directory {:?}", samples_path);
                std::fs::create_dir_all(samples_path.to_str().unwrap())?;
            }

            let sketchbook_path = proj_dirs.config_dir().join("sketchbook");
            if !sketchbook_path.exists() {
                println!("create megra sketchbook directory {:?}", sketchbook_path);
                std::fs::create_dir_all(sketchbook_path.to_str().unwrap())?;
            }

            println!("load samples from path: {:?}", samples_path);
            let ruffbox2 = sync::Arc::clone(&ruffbox);
            let sample_set2 = sync::Arc::clone(&sample_set);
            thread::spawn(move || {
                commands::load_sample_sets_path(&ruffbox2, &sample_set2, &samples_path);
                println!("a command (load default sample sets)");
            });
        }
    }

    if editor {
        editor::run_editor(
            &session,
            &ruffbox,
            &global_parameters,
            &sample_set,
            &parts_store,
            mode,
            font,
        );
        Ok(())
    } else {
        // star1t the megra repl
        repl::start_repl(
            &session,
            &ruffbox,
            &global_parameters,
            &sample_set,
            &parts_store,
            mode,
        )
    }
}
