use std::collections::HashMap;
use std::io;
use anyhow::{Result};
use pulldown_cmark::{Alignment, CodeBlockKind, CowStr, Event, LinkType, Options, Parser, Tag};
use pulldown_cmark::escape::{escape_href, escape_html, StrWrite};

enum TableState {
    Head,
    Body,
}

struct HtmlWriter<'a, I, W> {
    iter: I,
    writer: W,
    end_newline: bool,
    table_state: TableState,
    table_alignments: Vec<Alignment>,
    table_cell_index: usize,
    numbers: HashMap<CowStr<'a>, usize>,
}

impl<'a, I, W> HtmlWriter<'a, I, W> where I: Iterator<Item=Event<'a>>, W: StrWrite {
    fn new(iter: I, writer: W) -> Self {
        Self {
            iter,
            writer,
            end_newline: true,
            table_state: TableState::Head,
            table_alignments: Vec::new(),
            table_cell_index: 0,
            numbers: HashMap::new(),
        }
    }

    fn write_newline(&mut self) -> io::Result<()> {
        self.end_newline = true;
        self.writer.write_str("\n")
    }

    #[inline]
    fn write(&mut self, s: &str) -> io::Result<()> {
        self.writer.write_str(s)?;

        if !s.is_empty() {
            self.end_newline = s.ends_with('\n');
        }
        Ok(())
    }

    fn run(mut self) -> io::Result<()> {
        while let Some(event) = self.iter.next() {
            match event {
                Event::Start(tag) => { self.start_tag(tag)?; }
                Event::End(tag) => { self.end_tag(tag)?; }
                Event::Text(text) => {
                    escape_html(&mut self.writer, &text)?;
                    self.end_newline = text.ends_with('\n');
                }
                Event::Code(text) => {
                    self.write("<code>")?;
                    escape_html(&mut self.writer, &text)?;
                    self.write("</code>")?;
                }
                Event::Html(html) => {
                    self.write(&html)?;
                }
                Event::SoftBreak => {
                    self.write_newline()?;
                }
                Event::HardBreak => {
                    self.write("<br />\n")?;
                }
                Event::Rule => {
                    if self.end_newline {
                        self.write("<hr />\n")?;
                    } else {
                        self.write("\n<hr />\n")?;
                    }
                }
                Event::FootnoteReference(name) => {
                    let len = self.numbers.len() + 1;
                    self.write("<sup class=\"footnote-reference\"><a href=\"#")?;
                    escape_html(&mut self.writer, &name)?;
                    self.write("\">")?;
                    let number = *self.numbers.entry(name).or_insert(len);
                    write!(self.writer, "{}", number)?;
                    self.write("</a></sup>")?;
                }
                Event::TaskListMarker(true) => {
                    self.write("<input disabled=\"\" type=\"checkbox\" checked=\"\"/>\n")?;
                }
                Event::TaskListMarker(false) => {
                    self.write("<input disabled=\"\" type=\"checkbox\"/>\n")?;
                }
            }
        }
        Ok(())
    }

    fn start_tag(&mut self, tag: Tag<'a>) -> io::Result<()> {
        match tag {
            Tag::Paragraph => {
                if self.end_newline {
                    self.write("<p>")
                } else {
                    self.write("\n<p>")
                }
            }
            Tag::Heading(level, id, classes) => {
                if self.end_newline {
                    self.end_newline = false;
                    self.write("<")?;
                } else {
                    self.write("\n<")?;
                }
                write!(self.writer, "{}", level)?;
                if let Some(id) = id {
                    self.write(" id=\"")?;
                    escape_html(&mut self.writer, id)?;
                    self.write("\"")?;
                }
                let mut classes = classes.iter();
                if let Some(class) = classes.next() {
                    self.write(" class=\"")?;
                    escape_html(&mut self.writer, class)?;
                    for class in classes {
                        self.write(" ")?;
                        escape_html(&mut self.writer, class)?;
                    }
                    self.write("\"")?;
                }
                self.write(">")
            }
            Tag::Table(alignments) => {
                self.table_alignments = alignments;
                self.write("<table>")
            }
            Tag::TableHead => {
                self.table_state = TableState::Head;
                self.table_cell_index = 0;
                self.write("<thead><tr>")
            }
            Tag::TableRow => {
                self.table_cell_index = 0;
                self.write("<tr>")
            }
            Tag::TableCell => {
                match self.table_state {
                    TableState::Head => {
                        self.write("<th")?;
                    }
                    TableState::Body => {
                        self.write("<td")?;
                    }
                }
                match self.table_alignments.get(self.table_cell_index) {
                    Some(&Alignment::Left) => self.write(" style=\"text-align: left\">"),
                    Some(&Alignment::Center) => self.write(" style=\"text-align: center\">"),
                    Some(&Alignment::Right) => self.write(" style=\"text-align: right\">"),
                    _ => self.write(">"),
                }
            }
            Tag::BlockQuote => {
                if self.end_newline {
                    self.write("<blockquote>\n")
                } else {
                    self.write("\n<blockquote>\n")
                }
            }
            Tag::CodeBlock(info) => {
                if !self.end_newline {
                    self.write_newline()?;
                }
                match info {
                    CodeBlockKind::Fenced(info) => {
                        let lang = info.split(' ').next().unwrap();
                        if lang.is_empty() {
                            self.write("<pre><code>")
                        } else {
                            self.write("<pre><code class=\"language-")?;
                            escape_html(&mut self.writer, lang)?;
                            self.write("\">")
                        }
                    }
                    CodeBlockKind::Indented => self.write("<pre><code>"),
                }
            }
            Tag::List(Some(1)) => {
                if self.end_newline {
                    self.write("<ol>\n")
                } else {
                    self.write("\n<ol>\n")
                }
            }
            Tag::List(Some(start)) => {
                if self.end_newline {
                    self.write("<ol start=\"")?;
                } else {
                    self.write("\n<ol start=\"")?;
                }
                write!(self.writer, "{}", start)?;
                self.write("\">\n")
            }
            Tag::List(None) => {
                if self.end_newline {
                    self.write("<ul>\n")
                } else {
                    self.write("\n<ul>\n")
                }
            }
            Tag::Item => {
                if self.end_newline {
                    self.write("<li>")
                } else {
                    self.write("\n<li>")
                }
            }
            Tag::Emphasis => self.write("<em>"),
            Tag::Strong => self.write("<strong>"),
            Tag::Strikethrough => self.write("<del>"),
            Tag::Link(LinkType::Email, dest, title) => {
                self.write("<a href=\"mailto:")?;
                escape_href(&mut self.writer, &dest)?;
                if !title.is_empty() {
                    self.write("\" title=\"")?;
                    escape_html(&mut self.writer, &title)?;
                }
                self.write("\">")
            }
            Tag::Link(_link_type, dest, title) => {
                self.write("<a href=\"")?;
                if dest.ends_with(".md") {
                    escape_href(&mut self.writer, &format!("{}.html", dest.strip_suffix(".md").unwrap()))?;
                } else {
                    escape_href(&mut self.writer, &dest)?;
                }
                if !title.is_empty() {
                    self.write("\" title=\"")?;
                    escape_html(&mut self.writer, &title)?;
                }
                self.write("\">")
            }
            Tag::Image(_link_type, dest, title) => {
                self.write("<img src=\"")?;
                escape_href(&mut self.writer, &dest)?;
                self.write("\" alt=\"")?;
                self.raw_text()?;
                if !title.is_empty() {
                    self.write("\" title=\"")?;
                    escape_html(&mut self.writer, &title)?;
                }
                self.write("\" />")
            }
            Tag::FootnoteDefinition(name) => {
                if self.end_newline {
                    self.write("<div class=\"footnote-definition\" id=\"")?;
                } else {
                    self.write("\n<div class=\"footnote-definition\" id=\"")?;
                }
                escape_html(&mut self.writer, &*name)?;
                self.write("\"><sup class=\"footnote-definition-label\">")?;
                let len = self.numbers.len() + 1;
                let number = *self.numbers.entry(name).or_insert(len);
                write!(self.writer, "{}", number)?;
                self.write("</sup>")
            }
        }
    }

    fn end_tag(&mut self, tag: Tag) -> io::Result<()> {
        match tag {
            Tag::Paragraph => {
                self.write("</p>\n")?;
            }
            Tag::Heading(level, _id, _classes) => {
                self.write("</")?;
                write!(self.writer, "{}", level)?;
                self.write(">\n")?;
            }
            Tag::Table(_) => {
                self.write("</tbody></table>\n")?;
            }
            Tag::TableHead => {
                self.write("</tr></thead><tbody>\n")?;
                self.table_state = TableState::Body;
            }
            Tag::TableRow => {
                self.write("</tr>\n")?;
            }
            Tag::TableCell => {
                match self.table_state {
                    TableState::Head => {
                        self.write("</th>")?;
                    }
                    TableState::Body => {
                        self.write("</td>")?;
                    }
                }
                self.table_cell_index += 1;
            }
            Tag::BlockQuote => {
                self.write("</blockquote>\n")?;
            }
            Tag::CodeBlock(_) => {
                self.write("</code></pre>\n")?;
            }
            Tag::List(Some(_)) => {
                self.write("</ol>\n")?;
            }
            Tag::List(None) => {
                self.write("</ul>\n")?;
            }
            Tag::Item => {
                self.write("</li>\n")?;
            }
            Tag::Emphasis => {
                self.write("</em>")?;
            }
            Tag::Strong => {
                self.write("</strong>")?;
            }
            Tag::Strikethrough => {
                self.write("</del>")?;
            }
            Tag::Link(_, _, _) => {
                self.write("</a>")?;
            }
            Tag::Image(_, _, _) => (), // shouldn't happen, handled in start
            Tag::FootnoteDefinition(_) => {
                self.write("</div>\n")?;
            }
        }
        Ok(())
    }

    fn raw_text(&mut self) -> io::Result<()> {
        let mut nest = 0;
        while let Some(event) = self.iter.next() {
            match event {
                Event::Start(_) => nest += 1,
                Event::End(_) => {
                    if nest == 0 { break; }
                    nest -= 1;
                }
                Event::Html(text) | Event::Code(text) | Event::Text(text) => {
                    escape_html(&mut self.writer, &text)?;
                    self.end_newline = text.ends_with('\n');
                }
                Event::SoftBreak | Event::HardBreak | Event::Rule => {
                    self.write(" ")?;
                }
                Event::FootnoteReference(name) => {
                    let len = self.numbers.len() + 1;
                    let number = *self.numbers.entry(name).or_insert(len);
                    write!(self.writer, "[{}]", number)?;
                }
                Event::TaskListMarker(true) => self.write("[x]")?,
                Event::TaskListMarker(false) => self.write("[ ]")?,
            }
        }
        Ok(())
    }
}

/// Parse Markdown
pub fn parse(content: &str) -> Result<String> {
    let mut options = Options::empty();
    options.insert(Options::ENABLE_STRIKETHROUGH);
    let parser = Parser::new_ext(content, options);
    let mut result = String::new();
    HtmlWriter::new(parser, &mut result).run()?;
    Ok(result)
}


#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn paragraph() {
        assert_eq!(
            parse("\
        hello\n\n\
        world").unwrap(),
            "<p>hello</p>\n<p>world</p>\n".to_string()
        );
    }

    #[test]
    fn heading() {
        assert_eq!(parse("#h1").unwrap(), "<p>#h1</p>\n".to_string());
        assert_eq!(parse("# h1").unwrap(), "<h1>h1</h1>\n".to_string());
        assert_eq!(parse("## h2").unwrap(), "<h2>h2</h2>\n".to_string());
        assert_eq!(parse("### h3").unwrap(), "<h3>h3</h3>\n".to_string());
    }

    #[test]
    fn thematic_breaks() {
        assert_eq!(parse("***\n---\n___").unwrap(), "<hr />\n<hr />\n<hr />\n".to_string());
        assert_eq!(parse("+++").unwrap(), "<p>+++</p>\n".to_string());
    }

    #[test]
    fn block_quote() {
        assert_eq!(parse("> quote").unwrap(), "<blockquote>\n<p>quote</p>\n</blockquote>\n".to_string());
        assert_eq!(parse(r#"
> quote1
> quote2"#).unwrap(), "<blockquote>\n<p>quote1\nquote2</p>\n</blockquote>\n".to_string());
    }

    #[test]
    fn code_bock() {
        assert_eq!(parse("some `var`").unwrap(), "<p>some <code>var</code></p>\n".to_string());
        assert_eq!(parse(r#"
```js
alert()
```"#).unwrap(), "<pre><code class=\"language-js\">alert()\n</code></pre>\n".to_string());
    }

    #[test]
    fn list() {
        assert_eq!(parse(r#"
* 123
* 456"#).unwrap(), "<ul>\n<li>123</li>\n<li>456</li>\n</ul>\n".to_string());

        assert_eq!(parse(r#"
- 123
- 456"#).unwrap(), "<ul>\n<li>123</li>\n<li>456</li>\n</ul>\n".to_string());

        assert_eq!(parse(r#"
1. 123
2. 456"#).unwrap(), "<ol>\n<li>123</li>\n<li>456</li>\n</ol>\n".to_string());

        assert_eq!(parse(r#"
3. 123
4. 456"#).unwrap(), "<ol start=\"3\">\n<li>123</li>\n<li>456</li>\n</ol>\n".to_string());
    }

    #[test]
    fn emphasis() {
        assert_eq!(parse(r#"**BOLD**"#).unwrap(), "<p><strong>BOLD</strong></p>\n".to_string());
    }

    #[test]
    fn strikethrough() {
        assert_eq!(parse(r#"~~DEL~~"#).unwrap(), "<p><del>DEL</del></p>\n".to_string());
    }

    #[test]
    fn link() {
        assert_eq!(parse(r#"[Link](https://github.com)"#).unwrap(), "<p><a href=\"https://github.com\">Link</a></p>\n".to_string());
        assert_eq!(parse(r#"<i@tmk.im>"#).unwrap(), "<p><a href=\"mailto:i@tmk.im\">i@tmk.im</a></p>\n".to_string());
        assert_eq!(parse(r#"[Chapter 01](./chapter01)"#).unwrap(), "<p><a href=\"./chapter01\">Chapter 01</a></p>\n".to_string());
        assert_eq!(parse(r#"[Chapter 01](./chapter01.md)"#).unwrap(), "<p><a href=\"./chapter01.html\">Chapter 01</a></p>\n".to_string());
        assert_eq!(parse(r#"[Chapter 01](./chapter01.html)"#).unwrap(), "<p><a href=\"./chapter01.html\">Chapter 01</a></p>\n".to_string());
    }

    #[test]
    fn image() {
        assert_eq!(parse(r#"![Image](https://github.com)"#).unwrap(), "<p><img src=\"https://github.com\" alt=\"Image\" /></p>\n".to_string());
    }
}