use std::{env, fs};
use std::path::{Path, PathBuf};
use ansi_term::Color;
use anyhow::{Result};
use glob::{glob};
use crate::parser::parse;
use crate::utils::normalize;

#[derive(Debug)]
pub struct GenerateArgs {
    pub dir: String,
    pub output: String,
}

/// Generate doc site
pub fn generate(args: &GenerateArgs) -> Result<()> {
    let current_dir = env::current_dir().unwrap();
    let dir = Path::new(&current_dir).join(&args.dir).canonicalize().expect("dir does not exist");
    let output = normalize(&Path::new(&current_dir).join(&args.output));

    log::debug!("dir: {:?}", dir);
    log::debug!("output: {:?}", output);

    if !output.starts_with(&dir) {
        println!("{}", Color::Yellow.paint("output dir is at the outside of input dir"));
    }

    let pattern = dir.join("**/*.md");
    let entries: Vec<PathBuf> = glob(pattern.to_str().unwrap())
        .unwrap()
        .filter_map(|p| p.ok())
        .collect();
    log::info!("entries: {:#?}", entries);

    if entries.is_empty() {
        println!("{}", Color::Yellow.paint(format!("There are no Markdown files in {:?}.", dir)));
        return Ok(());
    }
    fs::create_dir_all(&output).expect("Create output dir failed");

    for entry in entries {
        println!("Processing {}", Color::Blue.paint(entry.to_str().unwrap()));
        let content = fs::read(&entry)?;
        let content = String::from_utf8(content)?;
        let result = parse(&content).expect("Parse markdown failed");
        let filename = entry.strip_prefix(&dir)?.with_extension("html");
        let target_file = output.join(&filename);
        log::debug!("Save to: {:?}", target_file);
        if !target_file.parent().unwrap().exists() {
            fs::create_dir_all(target_file.parent().unwrap()).expect("Create dir failed");
        }
        fs::write(target_file, result).expect("Write failed");
    }

    Ok(())
}
