use rkyv::{ser::serializers::AllocSerializer, Deserialize, Infallible, Serialize};

use crate::path_builder::PathBuilder;
use crate::DataFile;
use crate::Error;
use crate::Frame;
use crate::Tick;

use crate::index::{Trie, TrieCommon};

pub type FrameIndex<T: Tick> = Trie<u64, T>;

/// A Epoch is an indexed time ordered chunk of data
pub struct Epoch<T: Tick> {
    /// Frame Set Holds the data for this epoch
    frame_index_backing: DataFile<FrameIndex<T>>,
    frame_index: FrameIndex<T>,

    /// Epoch Number
    epoch: u64,

    tainted: bool,

    /// Storage Folder Structure
    path_builder: PathBuilder,
}

impl<T: Tick> Epoch<T>
where
    T: Serialize<AllocSerializer<4096>>,
    T::Archived: Deserialize<T, Infallible>,
{
    #[inline(always)]
    pub fn new(epoch: u64, path_builder: PathBuilder) -> Result<Epoch<T>, Error> {
        let mut frame_index_backing =
            DataFile::<FrameIndex<T>>::new(path_builder.index_backing_file(epoch))?;

        let frame_index = frame_index_backing
            .try_read()
            .unwrap_or_else(|_| Trie::new());

        Ok(Epoch {
            frame_index_backing,
            frame_index,
            epoch,
            tainted: false,
            path_builder: path_builder.clone(),
        })
    }

    pub fn frames(&mut self) -> impl Iterator<Item = Frame<T>> + '_ {
        self.frame_index
            .iter()
            .map(|(time, item)| Frame::new(*time, item.clone()))
    }

    #[inline(always)]
    pub fn insert(&mut self, frame: &Frame<T>) -> Result<(), Error>
    where
        T: Serialize<AllocSerializer<4096>>,
    {
        let time = frame.time();

        if self.frame_index.get(&time).is_some() {
            return Err(Error::FrameConflict);
        }

        self.frame_index.insert(time, frame.tick().clone());

        self.tainted = true;

        Ok(())
    }

    #[inline(always)]
    pub fn epoch(&self) -> u64 {
        self.epoch
    }

    #[inline(always)]
    pub fn persist(&mut self) -> Result<(), Error> {
        if !self.tainted {
            return Ok(());
        }

        self.frame_index_backing.write_all(&self.frame_index)?;

        self.tainted = false;

        Ok(())
    }
}

// impl<T: Tick> Drop for Epoch<T> {
//     fn drop(&mut self) {
//         //self.persist();
//     }
// }

// impl<T: Tick> Drop for Epoch<T>
// where
//     T: Serialize<AllocSerializer<4096>>,
//     T::Archived: Deserialize<T, Infallible>,
// {
//     fn drop(&mut self) {
//         self.persist();
//     }
// }
