//! Magma DB

use std::path::Path;

use crate::{path_builder::PathBuilder, DataFile, Epoch, Error, Frame, Result, Tick};

use rkyv::{ser::serializers::AllocSerializer, Archive, Deserialize, Infallible, Serialize};
use std::marker::PhantomData;
use std::slice::Iter;

pub fn init_paths(path_builder: &PathBuilder) -> std::io::Result<()> {
    std::fs::create_dir_all(path_builder.epoch_path())
}

pub struct Magma<T: Tick> {
    /// Asset Name
    asset: String,
    /// Source Venue, eg Exchange Name
    source: Option<String>,

    epoch_index_backing: DataFile<Vec<u64>>,
    pub(crate) epoch_index: Vec<u64>,
    curr_epoch: (u64, Option<Epoch<T>>),

    /// Storage Folder Structure
    path_builder: PathBuilder,

    _phantom: PhantomData<T>,
}

impl<T: Tick> Magma<T>
where
    T: Serialize<AllocSerializer<4096>>,
    T::Archived: Deserialize<T, Infallible>,
{
    /// Create a new Tick Datastore
    pub fn new(asset: &str, base_path: impl AsRef<Path>) -> Result<Magma<T>> {
        let path_builder = PathBuilder::new(&asset, base_path);

        init_paths(&path_builder)?;

        let mut epoch_index_backing =
            DataFile::<Vec<u64>>::new(path_builder.epoch_index_backing_file())?;

        let epoch_index = epoch_index_backing
            .try_read()
            .unwrap_or_else(|_| Vec::new());

        Ok(Magma {
            asset: asset.to_string(),
            source: None,
            epoch_index_backing,
            epoch_index,
            curr_epoch: (0, None),
            path_builder: path_builder,
            _phantom: PhantomData,
        })
    }

    /// Insert new frame into storage
    #[inline(always)]
    pub fn insert(&mut self, frame: &Frame<T>) -> Result<()>
    where
        T: Serialize<AllocSerializer<4096>>,
    {
        let frame_epoch = frame.epoch();

        if self.needs_epoch_update(frame_epoch) {
            self.load_epoch(frame_epoch)?;
        }

        let curr_epoch = &mut self.curr_epoch;

        let ref mut frame_set = curr_epoch.1.as_mut().ok_or(Error::BadFrameTick)?;

        frame_set.insert(frame)?;

        Ok(())
    }

    #[inline(always)]
    fn needs_epoch_update(&self, epoch: u64) -> bool {
        let curr_epoch = &self.curr_epoch;

        let epoch_mismatch = epoch != curr_epoch.0;
        let need_epoch = curr_epoch.1.is_none();

        epoch_mismatch || need_epoch
    }

    #[inline(always)]
    pub fn load_epoch(&mut self, epoch: u64) -> Result<()> {
        self.curr_epoch = (epoch, Some(Epoch::new(epoch, self.path_builder.clone())?));

        self.insert_epoch(epoch)?;

        Ok(())
    }

    #[inline(always)]
    pub fn insert_epoch(&mut self, epoch: u64) -> Result<()> {
        let epoch_index = &mut self.epoch_index;

        let bin_search = epoch_index.binary_search(&epoch);

        match bin_search {
            Ok(_) => {} // already exists
            Err(pos) => {
                epoch_index.insert(pos, epoch);
            }
        }

        Ok(())
    }

    /// Read the last X amount of entries
    pub async fn read_last(depth: u64) {}

    /// Flush Data to storage
    pub fn persist(&mut self) -> Result<()> {
        let epoch_index = &mut self.epoch_index;
        let curr_epoch = &mut self.curr_epoch;

        self.epoch_index_backing.write_all(&epoch_index)?;

        if let Some(ref mut epoch) = curr_epoch.1 {
            epoch.persist()?;
        }

        Ok(())
    }

    /// Get asset name
    pub fn asset(&self) -> String {
        self.asset.clone()
    }

    #[inline(always)]
    pub fn epochs(&self) -> EpochIter<T> {
        EpochIter::<T>::new(self.epoch_index.iter(), self.path_builder.clone())
    }
}

impl<T: Tick> Drop for Magma<T> {
    fn drop(&mut self) {
        //self.persist();
    }
}

pub struct EpochIter<'a, T: Tick> {
    epoch_iter: Iter<'a, u64>,
    curr_epoch: Option<Epoch<T>>,
    path_builder: PathBuilder,
}

impl<'a, T: Tick> EpochIter<'a, T>
where
    T: Serialize<AllocSerializer<4096>>,
    T::Archived: Deserialize<T, Infallible>,
{
    #[inline(always)]
    pub fn new(epoch_iter: Iter<'a, u64>, path_builder: PathBuilder) -> Self {
        EpochIter {
            epoch_iter,
            curr_epoch: None,
            path_builder,
        }
    }
}

impl<'a, T: 'a + Tick> Iterator for EpochIter<'a, T>
where
    T: Serialize<AllocSerializer<4096>>,
    T::Archived: Deserialize<T, Infallible>,
{
    type Item = Epoch<T>;

    #[inline(always)]
    fn next(&mut self) -> Option<Self::Item> {
        let epoch = *self.epoch_iter.next()?;

        Epoch::new(epoch, self.path_builder.clone()).ok()
    }
}
