use std::fs::{File, OpenOptions};
use std::io::{Read, Seek, SeekFrom, Write};
use std::marker::PhantomData;
use std::path::Path;

use rkyv::{
    archived_root,
    ser::{serializers::AllocSerializer, Serializer},
    Archive, Deserialize, Infallible, Serialize,
};

use crate::Error;
use memmap2::MmapOptions;

/// Core Storage File with random reads
pub struct DataFile<T> {
    file: File,
    _phantom: PhantomData<T>,
}

impl<T> DataFile<T> {
    /// Allocate a new file
    pub fn new(path: impl AsRef<Path>) -> Result<Self, Error> {
        // Set file permisions to read & write. Create new file if none exists
        // otherwise open file at given path.
        let file = OpenOptions::new()
            .read(true)
            .write(true)
            .create(true)
            .open(path.as_ref())?;

        Ok(DataFile {
            file,
            _phantom: PhantomData,
        })
    }

    /// Get the total bytes of data file
    pub fn file_size(&mut self) -> Result<u64, Error> {
        let metadata = self.file.metadata().map_err(|err| Error::IoError(err))?;

        Ok(metadata.len())
    }

    /// Truncates or extends the underlying file, updating the size of this file to become size.
    pub fn set_len(&mut self, new_len: u64) -> Result<(), Error> {
        self.file
            .set_len(new_len)
            .map_err(|err| Error::IoError(err))?;

        Ok(())
    }

    /// Delete all data in file, leads to data loss.
    pub fn truncate(&mut self) -> Result<(), Error> {
        self.set_len(0)
    }
}

impl<T: Archive> DataFile<T>
where
    T::Archived: Deserialize<T, Infallible>,
{
    /// Read Frame from given offset with length
    pub fn read(&mut self, offset: u64, size: u64) -> Result<T, Error> {
        // Move cursor to starting postion
        self.file.seek(SeekFrom::Start(offset))?;
        // Allocate vector with needed capacity
        let mut buf = Vec::with_capacity(size as usize);
        // force os to allocate memory
        buf.resize(size as usize, 0);
        // fill buffer with 0's
        buf.fill(0);

        // read from file
        self.file.read(&mut buf)?;
        // decode and return frame

        let archived_value = unsafe { archived_root::<T>(&buf) };

        let mut deserializer = Infallible::default();
        archived_value
            .deserialize(&mut deserializer)
            .map_err(|_| Error::RkyvDeserializer)
    }

    /// Try Reading Index from File
    pub fn try_read(&mut self) -> Result<T, Error> {
        let file_len = self.file_size()?;
        if file_len <= 0 {
            return Err(Error::DataFileEmpty);
        }

        // Move cursor to starting postion
        self.file.seek(SeekFrom::Start(0))?;

        let mmap = unsafe { MmapOptions::new().map(&self.file).unwrap() };
        //let decompressed_buf = decompress_to_vec(&buf).map_err(|err| Error::InflateError(err))?;

        let archived_value = unsafe { archived_root::<T>(mmap.as_ref()) };

        let mut deserializer = Infallible::default();
        archived_value
            .deserialize(&mut deserializer)
            .map_err(|_| Error::RkyvDeserializer)
    }
}

impl<T: Serialize<AllocSerializer<4096>>> DataFile<T> {
    /// Write Data to file starting at given position
    pub fn write(&mut self, position: SeekFrom, data: &T) -> Result<(u64, u64), Error>
    where
        T: Serialize<AllocSerializer<4096>>,
    {
        // Move cursor to starting postion
        let end_pos = self.file.seek(position)?;

        let mut serializer = AllocSerializer::<4096>::default();
        serializer.serialize_value(data).unwrap();
        // Serialize Data into raw bytes
        let buf = serializer.into_serializer().into_inner();

        // Write to file
        self.file.write(&buf)?;

        Ok((end_pos, buf.len() as u64))
    }

    /// Write new item to the end of the file and return (end position, data bytes len)
    pub fn append(&mut self, item: &T) -> Result<(u64, u64), Error>
    where
        T: Serialize<AllocSerializer<4096>>,
    {
        self.write(SeekFrom::End(0), item)
    }

    /// Write to file
    pub fn write_all(&mut self, item: &T) -> Result<(), Error>
    where
        T: Serialize<AllocSerializer<4096>>,
    {
        // Move cursor to starting postion
        self.file.seek(SeekFrom::Start(0))?;

        let mut serializer = AllocSerializer::<4096>::default();
        serializer.serialize_value(item).unwrap();
        let buf = serializer.into_serializer().into_inner();

        //let compressed_buf = compress_to_vec(&buf, 3);

        // flush to disk
        self.file.write(&buf)?;

        Ok(())
    }
}
