use criterion::{criterion_group, criterion_main, Criterion};
use mdb::index::Trie;
use rkyv::{
    archived_root,
    ser::{serializers::AllocSerializer, Serializer},
    Deserialize, Infallible,
};

fn get_text() -> Vec<String> {
    use std::fs::File;
    use std::io::Read;
    const DATA: &[&str] = &["data/1984.txt", "data/sun-rising.txt"];
    let mut contents = String::new();
    File::open(&DATA[1])
        .unwrap()
        .read_to_string(&mut contents)
        .unwrap();
    contents
        .split(|c: char| c.is_whitespace())
        .map(|s| s.to_string())
        .collect()
}

fn make_trie(words: &[String]) -> Trie<&str, usize> {
    let mut trie = Trie::new();
    for w in words {
        trie.insert(&w[..], w.len());
    }
    trie
}

fn trie_insert(b: &mut Criterion) {
    let words = get_text();
    b.bench_function("trie insert", |b| b.iter(|| make_trie(&words)));
}

fn trie_get(b: &mut Criterion) {
    let words = get_text();
    let trie = make_trie(&words);
    b.bench_function("trie get", |b| {
        b.iter(|| {
            words
                .iter()
                .map(|w| trie.get(&&w[..]))
                .collect::<Vec<Option<&usize>>>()
        })
    });
}

fn trie_insert_remove(b: &mut Criterion) {
    let words = get_text();

    b.bench_function("trie remove", |b| {
        b.iter(|| {
            let mut trie = make_trie(&words);
            for w in &words {
                trie.remove(&&w[..]);
            }
        });
    });
}

fn trie_ser(b: &mut Criterion) {
    let mut trie = Trie::new();
    trie.insert(1u64, 19u32);
    trie.insert(12u64, 35u32);
    trie.insert(113u64, 1u32);
    trie.insert(7u64, 5u32);

    b.bench_function("trie ser", |b| {
        b.iter(|| {
            let mut serializer = AllocSerializer::<4096>::default();
            serializer.serialize_value(&trie).unwrap();
            let _bytes = serializer.into_serializer().into_inner();
        });
    });
}

fn trie_der(b: &mut Criterion) {
    let mut trie = Trie::new();
    trie.insert(1u64, 19u32);
    trie.insert(12u64, 35u32);
    trie.insert(113u64, 1u32);
    trie.insert(7u64, 5u32);

    let mut serializer = AllocSerializer::<4096>::default();
    serializer.serialize_value(&trie).unwrap();
    let bytes = serializer.into_serializer().into_inner();

    b.bench_function("trie der", |b| {
        b.iter(|| {
            let archived = unsafe { archived_root::<Trie<u64, u32>>(&bytes[..]) };
            let _deserialized: Trie<u64, u32> = archived.deserialize(&mut Infallible).unwrap();
            //assert_eq!(deserialized, trie);
        });
    });
}

criterion_group!(
    benches,
    trie_insert,
    trie_get,
    trie_insert_remove,
    trie_ser,
    trie_der
);

criterion_main!(benches);
