use std::io;
use std::net::ToSocketAddrs;
use mco::net::TcpStream;
use crate::bytes::{ByteString};
use crate::client::Client;
use crate::cmd;
use crate::simple::SimpleClient;
use super::errors::ConnectError;

/// Redis connector
pub struct RedisConnector<A> {
    address: A,
    passwords: Vec<ByteString>,
}

impl<A> RedisConnector<A>
    where
        A: ToSocketAddrs + Clone,
{
    /// Create new redis connector
    pub fn new(address: A) -> RedisConnector<A> {
        RedisConnector {
            address: address.clone(),
            passwords: Vec::new(),
        }
    }
}

impl<A> RedisConnector<A>
    where
        A: ToSocketAddrs + Clone,
{
    /// Add redis auth password
    pub fn password<U>(mut self, password: U) -> Self
        where
            U: AsRef<str>,
    {
        self.passwords.push(ByteString::from(password.as_ref().to_string()));
        self
    }

    /// Use custom connector
    pub fn connector(self) -> RedisConnector<A> {
        RedisConnector {
            address: self.address,
            passwords: self.passwords,
        }
    }
}

impl<A> RedisConnector<A>
    where
        A: ToSocketAddrs + Clone,
{
    fn _connect(&mut self) -> Result<SimpleClient, ConnectError> {
        let passwords = self.passwords.clone();
        let conn = TcpStream::connect(self.address.clone())?;
        // io.set_memory_pool(pool);
        if passwords.is_empty() {
            Ok(SimpleClient::new(conn))
        } else {
            let client = SimpleClient::new(conn);
            for password in passwords {
                if client.exec(cmd::Auth(password))? {
                    return Ok(client);
                }
            }
            Err(ConnectError::Unauthorized)
        }
    }

    /// Connect to redis server and create shared client
    pub fn connect(&mut self) -> Result<Client, ConnectError> {
        Ok(Client::new(self._connect()?))
    }

    /// Connect to redis server and create simple client
    pub fn connect_simple(&mut self) -> Result<SimpleClient, ConnectError> {
        Ok(self._connect()?)
    }
}
