use super::{helpers, publish, CurrentOrders};
use crate::{
  config::{get_soft_reject_count_limit, get_x_death_count_limit},
  message_exchange::{
    rabbitmq::current_orders::{OrderDelivery, OrderDeliveryType},
    OrderMessage,
  },
  MessageError, Result,
};
use amq_protocol_types::FieldTable;
use async_std::{
  channel::Sender,
  future::timeout,
  stream::StreamExt,
  task::{self, JoinHandle},
};
use lapin::{
  message::Delivery,
  options::{BasicCancelOptions, BasicConsumeOptions, BasicNackOptions, BasicRejectOptions},
  Channel,
};
use std::{
  convert::TryFrom,
  sync::{
    atomic::{AtomicBool, Ordering},
    Arc, Mutex,
  },
  time::Duration,
};

pub struct RabbitmqConsumer {
  handle: Option<JoinHandle<()>>,
  should_consume: Arc<AtomicBool>,
  consumer_triggers: Arc<Mutex<Vec<Arc<AtomicBool>>>>,
}

pub const RABBITMQ_CONSUMER_TAG_JOB: &str = "amqp_worker";
pub const RABBITMQ_CONSUMER_TAG_DIRECT: &str = "status_amqp_worker";

const CONSUMER_TIMEOUT: Duration = Duration::from_millis(500);

impl RabbitmqConsumer {
  pub async fn new(
    channel: &Channel,
    sender: Sender<OrderMessage>,
    queue_name: &str,
    consumer_tag: &str,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<Self> {
    log::debug!("Start RabbitMQ consumer on queue {:?}", queue_name);

    let channel = Arc::new(channel.clone());

    let should_consume = Arc::new(AtomicBool::new(true));
    let consumer_triggers = Arc::new(Mutex::new(vec![]));

    let handle = Self::start_consumer(
      channel,
      queue_name,
      consumer_tag,
      sender,
      current_orders,
      should_consume.clone(),
      consumer_triggers.clone(),
    )
    .await?;

    Ok(RabbitmqConsumer {
      handle,
      should_consume,
      consumer_triggers,
    })
  }

  pub fn connect(&mut self, rabbitmq_consumer: &RabbitmqConsumer) {
    self
      .consumer_triggers
      .lock()
      .unwrap()
      .push(rabbitmq_consumer.get_trigger());
  }

  fn get_trigger(&self) -> Arc<AtomicBool> {
    self.should_consume.clone()
  }

  async fn start_consumer(
    channel: Arc<Channel>,
    queue_name: &str,
    consumer_tag: &str,
    sender: Sender<OrderMessage>,
    current_orders: Arc<Mutex<CurrentOrders>>,
    should_consume: Arc<AtomicBool>,
    consumer_triggers: Arc<Mutex<Vec<Arc<AtomicBool>>>>,
  ) -> Result<Option<JoinHandle<()>>> {
    let mut optional_consumer = Some(
      channel
        .basic_consume(
          queue_name,
          consumer_tag,
          BasicConsumeOptions::default(),
          FieldTable::default(),
        )
        .await?,
    );

    let queue_name = queue_name.to_string();
    let consumer_tag = consumer_tag.to_string();

    let handle = task::spawn(async move {
      loop {
        match (&optional_consumer, should_consume.load(Ordering::Relaxed)) {
          (Some(_), false) => {
            // if should not consume, unregister consumer from channel
            log::debug!("{} consumer unregisters from channel...", queue_name);

            optional_consumer = None;

            channel
              .basic_cancel(&consumer_tag, BasicCancelOptions::default())
              .await
              .map_err(MessageError::Amqp)
              .unwrap();
          }
          (None, true) => {
            // if should consume, reset channel consumer
            log::debug!("{} consumer resume consuming channel...", queue_name);

            optional_consumer = Some(
              channel
                .basic_consume(
                  &queue_name,
                  &consumer_tag,
                  BasicConsumeOptions::default(),
                  FieldTable::default(),
                )
                .await
                .unwrap(),
            );
          }
          _ => {}
        }

        if let Some(mut consumer) = optional_consumer.clone() {
          // Consume messages with timeout
          let next_delivery = timeout(CONSUMER_TIMEOUT, consumer.next()).await;

          if let Ok(Some(delivery)) = next_delivery {
            let (_, delivery) = delivery.expect("error in consumer");

            if !should_consume.load(Ordering::Relaxed) {
              // if should have not consumed a message, reject it and unregister from channel

              log::warn!(
                "{} consumer nacks and requeues received message, and unregisters from channel...",
                queue_name
              );

              optional_consumer = None;

              channel
                .basic_nack(
                  delivery.delivery_tag,
                  BasicNackOptions {
                    requeue: true,
                    ..Default::default()
                  },
                )
                .await
                .unwrap();

              channel
                .basic_cancel(&consumer_tag, BasicCancelOptions::default())
                .await
                .map_err(MessageError::Amqp)
                .unwrap();
            } else {
              // else process received message
              if let Err(error) = Self::process_delivery(
                sender.clone(),
                channel.clone(),
                &delivery,
                &queue_name.clone(),
                current_orders.clone(),
                consumer_triggers.clone(),
              )
              .await
              {
                log::error!("RabbitMQ consumer: {:?}", error);
                if let Err(error) = publish::error(channel.clone(), vec![delivery], &error).await {
                  log::error!("Unable to publish response: {:?}", error);
                }
              }
            }
          }
        }
      }
    });

    Ok(Some(handle))
  }

  async fn process_delivery(
    sender: Sender<OrderMessage>,
    channel: Arc<Channel>,
    delivery: &Delivery,
    queue_name: &str,
    current_orders: Arc<Mutex<CurrentOrders>>,
    consumer_triggers: Arc<Mutex<Vec<Arc<AtomicBool>>>>,
  ) -> Result<()> {
    let (count, soft_count) = helpers::get_message_death_count(delivery);
    let count = count.unwrap_or(0);
    let soft_count = soft_count.unwrap_or(0);
    let message_data = std::str::from_utf8(&delivery.data).map_err(|e| {
      MessageError::RuntimeError(format!("unable to retrieve raw message: {:?}", e))
    })?;

    let mut order_message = OrderMessage::try_from(message_data)?;

    log::debug!(
      "RabbitMQ consumer on {:?} queue received message: {:?} (iteration: {}, delivery: {})",
      queue_name,
      order_message,
      count + soft_count,
      delivery.delivery_tag,
    );

    // Message TTL management
    // TTL is handled within SDK by checking if `x-death` count is above limits
    // BUT for the `requirements` reject, it increases the custom `soft_reject` count and not
    // x-death to distinguish the two, as you cannot both reject and modify headers in RMQ.

    // Case when worker crashed and requeued message with flag redelivered
    // Reject to DLX in order to increase `x-death` count
    if delivery.redelivered {
      log::debug!("Send redelivered message to DLX for x-death increment.");
      return Self::reject_delivery(channel, delivery.delivery_tag, false).await;
    // Check if `x-death` count or `soft_reject` count is above specified limits
    } else if count > get_x_death_count_limit() || soft_count > get_soft_reject_count_limit() {
      order_message = match order_message {
        OrderMessage::Job(job) => OrderMessage::ReachedExpiration(job),
        OrderMessage::StartProcess(job) => OrderMessage::ReachedExpiration(job),
        _ => order_message,
      };
      current_orders
        .lock()
        .unwrap()
        .orders
        .push(OrderDelivery::new(
          delivery.clone(),
          OrderDeliveryType::ReachedExpiration,
        ));
    } else {
      match order_message {
        OrderMessage::Job(_) => {
          Self::handle_job(channel, delivery, current_orders).await?;
        }
        OrderMessage::InitProcess(_) => {
          Self::handle_init_process(channel, delivery, current_orders).await?;
        }
        OrderMessage::StartProcess(_) => {
          Self::handle_start_process(channel, delivery, current_orders).await?;
        }
        OrderMessage::UpdateProcess(_) => {
          Self::handle_update_process(channel, delivery, current_orders).await?;
        }
        OrderMessage::StopProcess(_) => {
          return Self::handle_stop_process(channel, delivery, current_orders).await;
        }
        OrderMessage::StopWorker => {
          Self::handle_stop_worker(channel, delivery, current_orders).await?;
        }
        OrderMessage::Status => {
          Self::handle_status(channel, delivery, current_orders).await?;
        }
        OrderMessage::StopConsumingJobs => {
          Self::handle_stop_consuming_jobs(channel, delivery, current_orders, consumer_triggers)
            .await?;
        }
        OrderMessage::ResumeConsumingJobs => {
          Self::handle_resume_consuming_jobs(channel, delivery, current_orders, consumer_triggers)
            .await?;
        }
        OrderMessage::ReachedExpiration(_) => {
          Self::handle_reached_expiration(channel, delivery, current_orders).await?;
        }
      }
    }

    log::debug!(
      "RabbitMQ consumer on {:?} queue forwards the order message: {:?}",
      queue_name,
      order_message
    );

    sender.send(order_message.clone()).await.map_err(|e| {
      MessageError::RuntimeError(format!("unable to send {:?} order: {:?}", order_message, e))
    })?;

    log::debug!("Order message sent!");

    Ok(())
  }

  async fn handle_job(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_initializing = current_order_types.contains(&OrderDeliveryType::Init);
    let is_starting = current_order_types.contains(&OrderDeliveryType::Start);
    let has_job = current_order_types.contains(&OrderDeliveryType::Job);

    if is_initializing || is_starting || has_job {
      // Worker already processing
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(delivery.clone(), OrderDeliveryType::Job));

    Ok(())
  }
  async fn handle_init_process(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_initializing = current_order_types.contains(&OrderDeliveryType::Init);
    let has_job = current_order_types.contains(&OrderDeliveryType::Job);

    if is_initializing || has_job {
      // Worker already processing
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Init,
      ));

    Ok(())
  }
  async fn handle_start_process(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_starting = current_order_types.contains(&OrderDeliveryType::Start);

    if is_starting {
      // Worker already processing
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Start,
      ));
    Ok(())
  }
  async fn handle_stop_process(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_initializing = current_order_types.contains(&OrderDeliveryType::Init);
    let is_starting = current_order_types.contains(&OrderDeliveryType::Start);
    let is_stopping = current_order_types.contains(&OrderDeliveryType::Stop);
    let has_job = current_order_types.contains(&OrderDeliveryType::Job);

    if is_stopping {
      log::warn!("Worker process already stopping");
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    if !is_initializing && !is_starting && !has_job {
      log::warn!("No process to stop");
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    // Setting the stop to Some make the exchange to return true on is_stopped()
    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Stop,
      ));

    Ok(())
  }
  async fn handle_update_process(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_initializing = current_order_types.contains(&OrderDeliveryType::Init);
    let is_starting = current_order_types.contains(&OrderDeliveryType::Start);
    let is_stopping = current_order_types.contains(&OrderDeliveryType::Stop);
    let has_job = current_order_types.contains(&OrderDeliveryType::Job);

    if is_stopping {
      log::warn!("Worker process is stopping");
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    if !is_initializing && !is_starting && !has_job {
      log::warn!("No process to update");
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    // Setting the update to Some make the exchange to return true on is_updated()
    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Update,
      ));

    Ok(())
  }

  async fn handle_stop_worker(
    _channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_initializing = current_order_types.contains(&OrderDeliveryType::Init);
    let is_starting = current_order_types.contains(&OrderDeliveryType::Start);
    let is_stopping = current_order_types.contains(&OrderDeliveryType::Stop);
    let has_job = current_order_types.contains(&OrderDeliveryType::Job);

    if !is_stopping && is_initializing || is_starting || has_job {
      log::info!("Stopping process...");
      // Setting the stop to Some make the exchange to return true on is_stopped()
      current_orders
        .lock()
        .unwrap()
        .orders
        .push(OrderDelivery::new(
          delivery.clone(),
          OrderDeliveryType::Stop,
        ));
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Kill,
      ));

    Ok(())
  }
  async fn handle_status(
    channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let current_order_types = current_orders.lock().unwrap().get_types();
    let is_checking_status = current_order_types.contains(&OrderDeliveryType::Status);

    if is_checking_status {
      // Worker already checking status
      return Self::reject_delivery(channel, delivery.delivery_tag, true).await;
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Status,
      ));

    Ok(())
  }
  async fn handle_stop_consuming_jobs(
    _channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
    consumer_triggers: Arc<Mutex<Vec<Arc<AtomicBool>>>>,
  ) -> Result<()> {
    // Stop consuming jobs queues
    for trigger in consumer_triggers.lock().unwrap().iter() {
      trigger.store(false, Ordering::Relaxed);
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Suspend,
      ));
    Ok(())
  }
  async fn handle_resume_consuming_jobs(
    _channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
    consumer_triggers: Arc<Mutex<Vec<Arc<AtomicBool>>>>,
  ) -> Result<()> {
    // Resume consuming jobs queues
    for trigger in consumer_triggers.lock().unwrap().iter() {
      trigger.store(true, Ordering::Relaxed);
    }

    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::Resume,
      ));

    Ok(())
  }

  async fn handle_reached_expiration(
    _channel: Arc<Channel>,
    delivery: &Delivery,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    current_orders
      .lock()
      .unwrap()
      .orders
      .push(OrderDelivery::new(
        delivery.clone(),
        OrderDeliveryType::ReachedExpiration,
      ));
    Ok(())
  }

  async fn reject_delivery(channel: Arc<Channel>, delivery_tag: u64, requeue: bool) -> Result<()> {
    log::warn!("Reject delivery {}", delivery_tag);
    channel
      .basic_reject(delivery_tag, BasicRejectOptions { requeue })
      .await
      .map_err(MessageError::Amqp)
  }
}

impl Drop for RabbitmqConsumer {
  fn drop(&mut self) {
    self.handle.take().map(JoinHandle::cancel);
  }
}
